package ru.yandex.solomon.alert.notification.channel.yachats;

import java.util.Map;
import java.util.concurrent.ScheduledExecutorService;

import javax.annotation.Nonnull;

import ru.yandex.solomon.alert.notification.channel.NotificationChannel;
import ru.yandex.solomon.alert.notification.channel.NotificationChannelFactory;
import ru.yandex.solomon.alert.notification.domain.Notification;
import ru.yandex.solomon.alert.notification.domain.NotificationType;
import ru.yandex.solomon.alert.notification.domain.telegram.TelegramNotification;
import ru.yandex.solomon.alert.notification.domain.yachats.YaChatsNotification;
import ru.yandex.solomon.alert.template.TemplateFactory;
import ru.yandex.solomon.alert.util.RateLimit;
import ru.yandex.solomon.alert.util.RateLimiters;
import ru.yandex.solomon.alert.yachats.YaChatsClient;

/**
 * @author Nikita Minin
 */
public class YaChatsNotificationChannelFactory implements NotificationChannelFactory {

    private final YaChatsClient yaChatsClient;
    private final TemplateFactory templateFactory;
    private final YaChatsTemplate template;
    private final YaChatsRawLimits rawLimits;
    private final ScheduledExecutorService scheduledExecutor;
    private final RateLimit generalRateLimit;

    public YaChatsNotificationChannelFactory(
        YaChatsClient yaChatsClient,
        TemplateFactory templateFactory,
        YaChatsTemplate template,
        YaChatsRawLimits rawLimits,
        ScheduledExecutorService scheduledExecutor)
    {
        this.yaChatsClient = yaChatsClient;
        this.templateFactory = templateFactory;
        this.template = template;
        this.rawLimits = rawLimits;
        this.scheduledExecutor = scheduledExecutor;
        this.generalRateLimit = RateLimiters.create(rawLimits.getGeneralRateLimit());
    }

    @Override
    public Map<NotificationType, Integer> getSupportTypes() {
        // We don't report TELEGRAM here, otherwise this factory would try to handle TELEGRAM channels
        return Map.of(NotificationType.YA_CHATS, 100);
    }

    private YaChatsNotificationChannel createYaChatsChannel(YaChatsNotification notification) {
        YaChatsTemplate template = this.template.override(templateFactory, notification);
        YaChatsLimits limits = YaChatsLimits.create(generalRateLimit, rawLimits);
        return new YaChatsNotificationChannel(
                notification,
                yaChatsClient,
                scheduledExecutor,
                limits,
                template);

    }

    private YaChatsNotificationChannel createTelegramFallbackChannel(TelegramNotification telegramNotification) {
        YaChatsNotification notification = new YaChatsNotification.Builder(telegramNotification).build();
        YaChatsTemplate template = this.template.override(templateFactory, notification);
        YaChatsLimits limits = YaChatsLimits.create(generalRateLimit, rawLimits);
        return new YaChatsNotificationChannel(
                notification,
                yaChatsClient,
                scheduledExecutor,
                limits,
                template);

    }

    @Nonnull
    @Override
    public NotificationChannel createChannel(@Nonnull Notification notification) {
        if (notification instanceof YaChatsNotification yaChatsNotification) {
            return createYaChatsChannel(yaChatsNotification);
        }
        if (notification instanceof TelegramNotification telegramNotification) {
            return createTelegramFallbackChannel(telegramNotification);
        }
        throw new IllegalArgumentException("Bad notification type " + notification.getType() + " for YaChats factory");
    }
}
