package ru.yandex.solomon.alert.notification.domain;

import java.time.Duration;
import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.EnumSet;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.concurrent.Immutable;
import javax.annotation.concurrent.NotThreadSafe;
import javax.annotation.concurrent.ThreadSafe;

import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.StringUtils;

import ru.yandex.solomon.alert.EvaluationStatus;
import ru.yandex.solomon.alert.dao.Entity;
import ru.yandex.solomon.alert.domain.AlertSeverity;
import ru.yandex.solomon.alert.domain.StringInterner;
import ru.yandex.solomon.util.collection.Nullables;

/**
 * @author Vladimir Gordiychuk
 */
@Immutable
@ThreadSafe
public abstract class Notification implements Entity {
    private static final Set<EvaluationStatus.Code> DEFAULT_NOTIFY_ABOUT_STATUS;
    static {
         DEFAULT_NOTIFY_ABOUT_STATUS = ImmutableSet.of(EvaluationStatus.Code.ALARM);
    }
    public static final Duration DEFAULT_REPEAT_NOTIFY_DELAY = Duration.ZERO;

    private final String id;
    private final String projectId;
    private final String folderId;
    private final String name;
    private final String description;

    /**
     * List of state that should trigger send notification.
     *
     * @see Notification#DEFAULT_NOTIFY_ABOUT_STATUS
     */
    private final Set<EvaluationStatus.Code> notifyAboutStatus;

    /**
     * Delay between repeated notify about the same evaluation status, by default notify about
     * particular evaluation status only after change it. Zero means that repeat not used
     *
     * @see Notification#DEFAULT_REPEAT_NOTIFY_DELAY
     */
    private final Duration repeatNotifyDelay;

    private final String createdBy;
    private final Instant createdAt;
    private final String updatedBy;
    private final Instant updatedAt;
    private final int version;
    private final boolean defaultForProject;
    private final Set<AlertSeverity> defaultForSeverity;
    private final Map<String, String> labels;

    protected Notification(Notification.Builder<?, ?> builder) {
        this.id = Objects.requireNonNull(builder.id, "id");
        this.projectId = Objects.requireNonNull(builder.projectId, "projectId");
        this.folderId = Nullables.orEmpty(builder.folderId);
        this.name = StringUtils.isEmpty(builder.name)
                ? builder.id
                : builder.name;
        this.defaultForProject = Nullables.orFalse(builder.defaultForProject);
        this.defaultForSeverity = Nullables.orEmpty(builder.defaultForSeverity);

        this.description = Nullables.orEmpty(builder.description);

        this.notifyAboutStatus = builder.notifyAboutStatus != null
                ? Sets.immutableEnumSet(builder.notifyAboutStatus)
                : DEFAULT_NOTIFY_ABOUT_STATUS;

        this.repeatNotifyDelay = builder.repeatNotifyDelay != null && !builder.repeatNotifyDelay.isNegative()
                ? builder.repeatNotifyDelay
                : DEFAULT_REPEAT_NOTIFY_DELAY;

        this.createdBy = Nullables.orEmpty(builder.createdBy);
        this.createdAt = builder.createdAt != null
                ? builder.createdAt
                : Instant.now();
        this.updatedBy = Nullables.orEmpty(builder.updatedBy);
        this.updatedAt = builder.updatedAt != null
                ? builder.updatedAt
                : Instant.now();
        this.version = builder.version;
        this.labels = ImmutableMap.copyOf(Nullables.orEmpty(builder.labels));
    }

    @Override
    public String getId() {
        return id;
    }

    @Override
    public String getProjectId() {
        return projectId;
    }

    public String getName() {
        return name;
    }

    public String getDescription() {
        return description;
    }

    @Override
    public String getFolderId() {
        return folderId;
    }

    public Set<EvaluationStatus.Code> getNotifyAboutStatus() {
        return notifyAboutStatus;
    }

    @Override
    public String getCreatedBy() {
        return createdBy;
    }

    @Override
    public String getUpdatedBy() {
        return updatedBy;
    }

    @Override
    public long getCreatedAt() {
        return createdAt.toEpochMilli();
    }

    @Override
    public long getUpdatedAt() {
        return updatedAt.toEpochMilli();
    }

    @Override
    public int getVersion() {
        return version;
    }

    public Map<String, String> getLabels() {
        return labels;
    }

    public Duration getRepeatNotifyDelay() {
        return repeatNotifyDelay;
    }

    public boolean isDefaultForProject() {
        return defaultForProject;
    }

    public Set<AlertSeverity> getDefaultForSeverity() {
        return defaultForSeverity;
    }

    public abstract NotificationType getType();

    public abstract Notification.Builder toBuilder();

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        Notification that = (Notification) o;

        return id.equals(that.id);
    }

    @Override
    public int hashCode() {
        return id.hashCode();
    }

    @Override
    public String toString() {
        return "Notification{" +
            "id='" + id + '\'' +
            ", projectId='" + projectId + '\'' +
            ", folderId='" + folderId + '\'' +
            ", name='" + name + '\'' +
            ", description='" + description + '\'' +
            ", notifyAboutStatus=" + notifyAboutStatus +
            ", repeatNotifyDelay=" + repeatNotifyDelay +
            ", labels=" + labels +
            ", createdBy='" + createdBy + '\'' +
            ", createdAt=" + createdAt +
            ", updatedBy='" + updatedBy + '\'' +
            ", updatedAt=" + updatedAt +
            ", version=" + version +
            ", defaultForProject=" + defaultForProject +
            ", defaultForSeverity=" + defaultForSeverity +
            '}';
    }

    @NotThreadSafe
    public static abstract class Builder<NotificationType extends Notification, BuilderType extends Notification.Builder<NotificationType, BuilderType>> {
        private String id;
        private String projectId;
        @Nullable
        private String folderId;
        @Nullable
        private String name;
        @Nullable
        private String description;
        @Nullable
        private Set<EvaluationStatus.Code> notifyAboutStatus;

        @Nullable
        private String createdBy;
        @Nullable
        private Instant createdAt;
        @Nullable
        private String updatedBy;
        @Nullable
        private Instant updatedAt;

        private int version = 0;

        @Nullable
        protected Duration repeatNotifyDelay;

        @Nullable
        private Map<String, String> labels;

        protected boolean defaultForProject;

        private Set<AlertSeverity> defaultForSeverity;

        protected Builder() {
        }

        public Builder(Notification notification) {
            this.id = notification.id;
            this.projectId = notification.projectId;
            this.folderId = notification.folderId;
            this.name = notification.name;
            this.notifyAboutStatus = notification.notifyAboutStatus;
            this.createdBy = notification.createdBy;
            this.createdAt = notification.createdAt;
            this.updatedBy = notification.updatedBy;
            this.updatedAt = notification.updatedAt;
            this.version = notification.version;
            this.repeatNotifyDelay = notification.repeatNotifyDelay;
            this.labels = ImmutableMap.copyOf(notification.labels);
            this.defaultForProject = notification.defaultForProject;
            this.defaultForSeverity = notification.defaultForSeverity;
        }

        protected abstract BuilderType self();

        /**
         * @see Notification#id
         */
        public BuilderType setId(String id) {
            this.id = StringInterner.I.intern(id);
            return self();
        }

        /**
         * @see Notification#projectId
         */
        public BuilderType setProjectId(String projectId) {
            this.projectId = StringInterner.I.intern(projectId);
            return self();
        }

        /**
         * @see Notification#folderId
         */
        public BuilderType setFolderId(String folderId) {
            this.folderId = StringInterner.I.intern(folderId);
            return self();
        }

        /**
         * @see Notification#name
         */
        public BuilderType setName(String name) {
            this.name = name;
            return self();
        }

        public BuilderType setDescription(String description) {
            this.description = description;
            return self();
        }

        /**
         * @see Notification#notifyAboutStatus
         */
        public BuilderType setNotifyAboutStatus(Set<EvaluationStatus.Code> status) {
            this.notifyAboutStatus = status;
            return self();
        }

        /**
         * @see Notification#notifyAboutStatus
         */
        public BuilderType setNotifyAboutStatus(EvaluationStatus.Code first, EvaluationStatus.Code... rest) {
            this.notifyAboutStatus = EnumSet.of(first, rest);
            return self();
        }

        /**
         * @see Notification#createdBy
         */
        public BuilderType setCreatedBy(String user) {
            this.createdBy = StringInterner.I.intern(user);
            return self();
        }

        /**
         * @see Notification#createdAt
         */
        public BuilderType setCreatedAt(Instant createdAt) {
            this.createdAt = createdAt.truncatedTo(ChronoUnit.SECONDS);
            return self();
        }

        public BuilderType setCreatedAt(long createdAt) {
            this.createdAt = Instant.ofEpochMilli(createdAt).truncatedTo(ChronoUnit.SECONDS);
            return self();
        }

        /**
         * @see Notification#updatedBy
         */
        public BuilderType setUpdatedBy(String user) {
            this.updatedBy = StringInterner.I.intern(user);
            return self();
        }

        /**
         * @see Notification#updatedAt
         */
        public BuilderType setUpdatedAt(Instant updatedAt) {
            this.updatedAt = updatedAt.truncatedTo(ChronoUnit.SECONDS);
            return self();
        }

        public BuilderType setUpdatedAt(long updatedAt) {
            this.updatedAt = Instant.ofEpochMilli(updatedAt).truncatedTo(ChronoUnit.SECONDS);
            return self();
        }

        /**
         * @see Notification#version
         */
        public BuilderType setVersion(int version) {
            this.version = version;
            return self();
        }

        public BuilderType setRepeatNotifyDelay(Duration delay) {
            this.repeatNotifyDelay = delay;
            return self();
        }

        public BuilderType setDefaultForProject(boolean defaultForProject) {
            this.defaultForProject = defaultForProject;
            return self();
        }

        public BuilderType setDefaultForSeverity(Set<AlertSeverity> defaultForSeverity) {
            this.defaultForSeverity = defaultForSeverity;
            return self();
        }

        public BuilderType setLabels(Map<String, String> labels) {
            var map = ImmutableMap.<String, String>builder();
            for (var entry : labels.entrySet()) {
                map.put(StringInterner.I.intern(entry.getKey()), StringInterner.I.intern(entry.getValue()));
            }
            this.labels = map.build();
            return self();
        }

        public BuilderType addLabel(String key, String value) {
            this.labels = ImmutableMap.<String, String>builder()
                    .putAll(labels)
                    .put(StringInterner.I.intern(key), StringInterner.I.intern(value))
                    .build();
            return self();
        }

        public abstract NotificationType build();
    }
}
