package ru.yandex.solomon.alert.notification.domain;

import javax.annotation.ParametersAreNonnullByDefault;
import javax.annotation.concurrent.NotThreadSafe;

import org.apache.commons.lang3.StringUtils;

import ru.yandex.solomon.alert.domain.StringInterner;


/**
 * @author Alexey Trushkin
 */
@ParametersAreNonnullByDefault
public class PhoneNotification extends Notification {

    private final String login;
    private final AbcDuty duty;

    private PhoneNotification(PhoneNotification.Builder builder) {
        super(builder);
        this.login = builder.login;
        this.duty = builder.duty;
        if (StringUtils.isEmpty(login)) {
            if (duty.dutySlug.isEmpty() || duty.abcService.isEmpty()) {
                throw new IllegalArgumentException("login or abc duty should be specified");
            }
        }
    }

    public static PhoneNotification.Builder newBuilder() {
        return new PhoneNotification.Builder();
    }

    @Override
    public NotificationType getType() {
        return NotificationType.PHONE_CALL;
    }

    @Override
    public PhoneNotification.Builder toBuilder() {
        return new PhoneNotification.Builder(this);
    }

    public String getLogin() {
        return login;
    }

    public AbcDuty getDuty() {
        return duty;
    }

    public String toDisplayText() {
        if (isLogin()) {
            return login;
        }
        return duty.abcService + "(" + duty.dutySlug + ")";
    }

    public boolean isLogin() {
        return !StringUtils.isEmpty(login);
    }

    @NotThreadSafe
    @ParametersAreNonnullByDefault
    public static class Builder extends Notification.Builder<PhoneNotification, PhoneNotification.Builder> {
        private String login = "";
        private AbcDuty duty = AbcDuty.EMPTY;

        private Builder() {
        }

        public Builder(PhoneNotification notification) {
            super(notification);
            this.login = notification.login;
            this.duty = notification.duty;
        }

        @Override
        protected PhoneNotification.Builder self() {
            return this;
        }

        /**
         * @see PhoneNotification#login
         */
        public PhoneNotification.Builder setLogin(String login) {
            this.login = StringInterner.I.intern(login);
            return self();
        }

        /**
         * @see PhoneNotification#duty
         */
        public PhoneNotification.Builder setDuty(AbcDuty duty) {
            this.duty = duty;
            return self();
        }

        @Override
        public PhoneNotification build() {
            return new PhoneNotification(this);
        }
    }

    public static record AbcDuty(String abcService, String dutySlug) {
        public static final AbcDuty EMPTY = new AbcDuty("", "");
    }
}
