package ru.yandex.solomon.alert.notification.domain.email;

import java.util.Collections;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;
import javax.annotation.concurrent.Immutable;
import javax.annotation.concurrent.NotThreadSafe;
import javax.annotation.concurrent.ThreadSafe;

import com.google.common.collect.ImmutableSet;

import ru.yandex.solomon.alert.notification.domain.Notification;
import ru.yandex.solomon.alert.notification.domain.NotificationType;

/**
 * @author Vladimir Gordiychuk
 */
@Immutable
@ThreadSafe
@ParametersAreNonnullByDefault
public class EmailNotification extends Notification {
    @Nullable
    private final String subjectTemplate;
    @Nullable
    private final String contentTemplate;
    private final Set<String> recipients;

    protected EmailNotification(Builder builder) {
        super(builder);
        this.subjectTemplate = builder.subjectTemplate;
        this.contentTemplate = builder.contentTemplate;
        this.recipients = builder.recipients;
    }

    public static EmailNotification.Builder newBuilder() {
        return new Builder();
    }

    @Override
    public NotificationType getType() {
        return NotificationType.EMAIL;
    }

    @Nullable
    public String getSubjectTemplate() {
        return subjectTemplate;
    }

    @Nullable
    public String getContentTemplate() {
        return contentTemplate;
    }

    public Set<String> getRecipients() {
        return recipients;
    }

    @Override
    public Builder toBuilder() {
        return new Builder(this);
    }

    @NotThreadSafe
    @ParametersAreNonnullByDefault
    public static class Builder extends Notification.Builder<EmailNotification, EmailNotification.Builder> {
        @Nullable
        private String subjectTemplate;
        @Nullable
        private String contentTemplate;
        private Set<String> recipients = Collections.emptySet();

        private Builder() {
        }

        public Builder(EmailNotification notification) {
            super(notification);
            this.subjectTemplate = notification.subjectTemplate;
            this.contentTemplate = notification.contentTemplate;
            this.recipients = notification.recipients;
        }

        @Override
        protected Builder self() {
            return this;
        }

        /**
         * @see EmailNotification#subjectTemplate
         */
        public Builder setSubjectTemplate(String template) {
            this.subjectTemplate = template;
            return self();
        }

        /**
         * @see EmailNotification#contentTemplate
         */
        public Builder setContentTemplate(String template) {
            this.contentTemplate = template;
            return self();
        }

        /**
         * @see EmailNotification#recipients
         */
        public Builder setRecipients(Iterable<String> recipients) {
            this.recipients = ImmutableSet.copyOf(recipients);
            return self();
        }

        /**
         * @see EmailNotification#recipients
         */
        public Builder setRecipient(String recipient) {
            this.recipients = Collections.singleton(recipient);
            return self();
        }

        @Override
        public EmailNotification build() {
            return new EmailNotification(this);
        }
    }
}
