package ru.yandex.solomon.alert.notification.domain.juggler;

import java.time.Duration;
import java.util.Collections;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;
import javax.annotation.concurrent.NotThreadSafe;

import com.google.common.collect.ImmutableList;

import ru.yandex.solomon.alert.notification.domain.Notification;
import ru.yandex.solomon.alert.notification.domain.NotificationType;

/**
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public class JugglerNotification extends Notification {
    /**
     * @see Notification#DEFAULT_REPEAT_NOTIFY_DELAY
     */
    private static final Duration DEFAULT_JUGGLER_REPEAT_NOTIFY_DELAY = Duration.ZERO;

    /**
     * FQDN that trigger notification, if not specified will be used virtual solomon-alert name.
     */
    private final String host;

    /**
     * Template for checking function. If not specified will be used alertId
     */
    private final String service;

    /**
     * Template for instance, if not specified will be used application-id
     */
    private final String instance;

    /**
     * Template for description, for example: There is no codes 5xx founded at all
     */
    private final String description;

    /**
     * List of tags to able find event in juggler api
     */
    private final List<String> tags;

    private JugglerNotification(Builder builder) {
        super(builder);
        this.host = builder.host;
        this.description = builder.description;
        this.service = builder.service;
        this.instance = builder.instance;
        this.tags = builder.tags;
    }

    public static Builder newBuilder() {
        return new Builder();
    }

    @Override
    public NotificationType getType() {
        return NotificationType.JUGGLER;
    }

    public String getHost() {
        return host;
    }

    public String getService() {
        return service;
    }

    public String getInstance() {
        return instance;
    }

    public String getJugglerDescription() {
        return description;
    }

    public List<String> getTags() {
        return tags;
    }

    @Override
    public Builder toBuilder() {
        return new Builder(this);
    }

    @NotThreadSafe
    @ParametersAreNonnullByDefault
    public static class Builder extends Notification.Builder<JugglerNotification, Builder> {
        private String host = "";
        private String service = "";
        private String description = "";
        private String instance = "";
        private List<String> tags = Collections.emptyList();

        private Builder() {
        }

        private Builder(JugglerNotification notification) {
            super(notification);
            this.host = notification.host;
            this.service = notification.service;
            this.instance = notification.instance;
            this.description = notification.description;
            this.tags = notification.tags;
        }

        @Override
        protected Builder self() {
            return this;
        }

        /**
         * @see JugglerNotification#host
         */
        public Builder setHost(String host) {
            this.host = host;
            return self();
        }

        /**
         * @see JugglerNotification#host
         */
        public Builder setService(String service) {
            this.service = service;
            return self();
        }

        /**
         * @see JugglerNotification#instance
         */
        public Builder setInstance(String instance) {
            this.instance = instance;
            return self();
        }

        /**
         * @see JugglerNotification#description
         */
        public Builder setJugglerDescription(String description) {
            this.description = description;
            return self();
        }

        /**
         * @see JugglerNotification#tags
         */
        public Builder setTags(Iterable<String> tags) {
            this.tags = ImmutableList.copyOf(tags);
            return self();
        }

        /**
         * @see JugglerNotification#tags
         */
        public Builder setTag(String tag) {
            this.tags = Collections.singletonList(tag);
            return self();
        }

        @Override
        public JugglerNotification build() {
            if (this.repeatNotifyDelay == null) {
                // by default juggler should be notified about each evaluation status
                // because without it juggler start fire NO_DATA for clients
                this.repeatNotifyDelay = DEFAULT_JUGGLER_REPEAT_NOTIFY_DELAY;
            }
            return new JugglerNotification(this);
        }
    }
}
