package ru.yandex.solomon.alert.notification.domain.sms;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;
import javax.annotation.concurrent.NotThreadSafe;

import org.apache.commons.lang3.StringUtils;

import ru.yandex.solomon.alert.notification.domain.Notification;
import ru.yandex.solomon.alert.notification.domain.NotificationType;
import ru.yandex.solomon.util.collection.Nullables;

import static com.google.common.base.Preconditions.checkArgument;

/**
 * @author Vladimir Gordiychuk
 */
public class SmsNotification extends Notification {
    private final String textTemplate;
    private final String phone;
    private final String login;

    protected SmsNotification(Builder builder) {
        super(builder);
        checkArgument(StringUtils.isNotEmpty(builder.phone) || StringUtils.isNotEmpty(builder.login), "Recipient not specified");
        this.textTemplate = Nullables.orEmpty(builder.textTemplate);
        this.phone = builder.phone;
        this.login = builder.login;
    }

    public static Builder newBuilder() {
        return new Builder();
    }

    @Override
    public NotificationType getType() {
        return NotificationType.SMS;
    }

    @Override
    public Builder toBuilder() {
        return new Builder(this);
    }

    public String getTextTemplate() {
        return textTemplate;
    }

    public String getPhone() {
        return phone;
    }

    public String getLogin() {
        return login;
    }

    @NotThreadSafe
    @ParametersAreNonnullByDefault
    public static class Builder extends Notification.Builder<SmsNotification, SmsNotification.Builder> {
        @Nullable
        private String textTemplate;
        private String phone = "";
        private String login = "";

        public Builder() {
        }

        public Builder(SmsNotification notification) {
            super(notification);
            this.textTemplate = notification.textTemplate;
            this.phone = notification.getPhone();
            this.login = notification.getLogin();
        }

        @Override
        protected Builder self() {
            return this;
        }

        public Builder setTextTemplate(String textTemplate) {
            this.textTemplate = textTemplate;
            return this;
        }

        public Builder setPhone(String phone) {
            this.phone = phone;
            return this;
        }

        public Builder setLogin(String login) {
            this.login = login;
            return this;
        }

        @Override
        public SmsNotification build() {
            return new SmsNotification(this);
        }
    }
}
