package ru.yandex.solomon.alert.notification.domain.telegram;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;
import javax.annotation.concurrent.NotThreadSafe;

import ru.yandex.solomon.alert.notification.domain.Notification;
import ru.yandex.solomon.alert.notification.domain.NotificationType;
import ru.yandex.solomon.util.collection.Nullables;


/**
 * @author alexlovkov
 **/
public class TelegramNotification extends Notification {

    private final long chatId;
    /**
     * user's staff login
     */
    @Nullable
    private final String login;
    private final String textTemplate;
    private final boolean sendScreenshot;

    public TelegramNotification(Builder builder) {
        super(builder);
        this.chatId = builder.chatId;
        this.login = builder.login;
        this.textTemplate = Nullables.orEmpty(builder.textTemplate);
        this.sendScreenshot = builder.sendScreenshot;
    }

    public static TelegramNotification.Builder newBuilder() {
        return new TelegramNotification.Builder();
    }

    @Override
    public NotificationType getType() {
        return NotificationType.TELEGRAM;
    }

    @Override
    public Builder toBuilder() {
        return new TelegramNotification.Builder(this);
    }

    public long getChatId() {
        return chatId;
    }

    @Nullable
    public String getLogin() {
        return login;
    }

    public String getTextTemplate() {
        return textTemplate;
    }

    public boolean isSendScreenshot() {
        return sendScreenshot;
    }

    @Override
    public String toString() {
        return "TelegramNotification{" +
            "chatId=" + chatId +
            ", login='" + login + '\'' +
            ", textTemplate='" + textTemplate + '\'' +
            ", sendScreenshot='" + sendScreenshot + '\'' +
            "} " + super.toString();
    }

    @NotThreadSafe
    @ParametersAreNonnullByDefault
    public static class Builder extends Notification.Builder<TelegramNotification, TelegramNotification.Builder> {

        private long chatId;
        @Nullable
        private String login = "";
        private String textTemplate;
        private boolean sendScreenshot = false;

        public Builder() {
        }

        public Builder(TelegramNotification notification) {
            super(notification);
            this.chatId = notification.chatId;
            this.login = notification.login;
            this.textTemplate = notification.textTemplate;
            this.sendScreenshot = notification.sendScreenshot;
        }

        @Override
        protected Builder self() {
            return this;
        }

        @Override
        public TelegramNotification build() {
            return new TelegramNotification(this);
        }

        public Builder setChatId(long chatId) {
            this.chatId = chatId;
            return this;
        }

        public Builder setLogin(String login) {
            this.login = login;
            return this;
        }

        public Builder setTextTemplate(String textTemplate) {
            this.textTemplate = textTemplate;
            return this;
        }

        public Builder setSendScreenshot(boolean sendScreenshot) {
            this.sendScreenshot = sendScreenshot;
            return this;
        }
    }
}
