package ru.yandex.solomon.alert.notification.domain.webhook;

import java.util.Collections;
import java.util.Map;
import java.util.Objects;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;
import javax.annotation.concurrent.Immutable;
import javax.annotation.concurrent.NotThreadSafe;
import javax.annotation.concurrent.ThreadSafe;

import com.google.common.collect.ImmutableMap;

import ru.yandex.solomon.alert.notification.domain.Notification;
import ru.yandex.solomon.alert.notification.domain.NotificationType;

/**
 * @author Vladimir Gordiychuk
 */
@Immutable
@ThreadSafe
@ParametersAreNonnullByDefault
public final class WebhookNotification extends Notification {
    /**
     * Url that will be use to POST request
     */
    private final String url;
    /**
     * Additional headers that need inject to notification request
     */
    private final Map<String, String> headers;
    /**
     * FreeMarket template(<a href="http://freemarker.org">freemarker.org</a>)
     * that will be use to create request.
     */
    @Nullable
    private final String template;

    private WebhookNotification(Builder builder) {
        super(builder);
        this.url = Objects.requireNonNull(builder.url);
        this.headers = ImmutableMap.copyOf(builder.headers);
        this.template = builder.template;
    }

    @Nonnull
    public static Builder newBuilder() {
        return new Builder();
    }

    @Override
    public NotificationType getType() {
        return NotificationType.WEBHOOK;
    }

    @Override
    public Builder toBuilder() {
        return new Builder(this);
    }

    public String getUrl() {
        return url;
    }

    public Map<String, String> getHeaders() {
        return headers;
    }

    @Nullable
    public String getTemplate() {
        return template;
    }

    @NotThreadSafe
    @ParametersAreNonnullByDefault
    public static class Builder extends Notification.Builder<WebhookNotification, WebhookNotification.Builder> {
        @Nullable
        private String url;
        private Map<String, String> headers = Collections.emptyMap();
        @Nullable
        private String template;

        private Builder() {
        }

        public Builder(WebhookNotification notification) {
            super(notification);
            this.url = notification.url;
            this.headers = notification.headers;
            this.template = notification.template;
        }

        @Override
        protected Builder self() {
            return this;
        }

        /**
         * @see WebhookNotification#url
         */
        public Builder setUrl(String url) {
            this.url = url;
            return self();
        }

        /**
         * @see WebhookNotification#headers
         */
        public Builder setHeaders(Map<String, String> headers) {
            this.headers = headers;
            return self();
        }

        /**
         * @see WebhookNotification#template
         */
        public Builder setTemplate(String template) {
            this.template = template;
            return self();
        }

        @Override
        public WebhookNotification build() {
            return new WebhookNotification(this);
        }
    }
}
