package ru.yandex.solomon.alert.notification.domain.yachats;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;
import javax.annotation.concurrent.NotThreadSafe;

import org.apache.commons.lang3.StringUtils;

import ru.yandex.solomon.alert.notification.domain.Notification;
import ru.yandex.solomon.alert.notification.domain.NotificationType;
import ru.yandex.solomon.alert.notification.domain.telegram.TelegramNotification;
import ru.yandex.solomon.util.collection.Nullables;

/**
 * @author Nikita Minin
 */
public class YaChatsNotification extends Notification {

    /**
     * user's staff login
     */
    @Nullable
    private final String login;
    @Nullable
    private final String groupId;
    private final String textTemplate;

    public YaChatsNotification(YaChatsNotification.Builder builder) {
        super(builder);
        this.login = builder.login;
        this.groupId = builder.groupId;
        this.textTemplate = Nullables.orEmpty(builder.textTemplate);
    }

    public static YaChatsNotification.Builder newBuilder() {
        return new YaChatsNotification.Builder();
    }

    @Override
    public NotificationType getType() {
        return NotificationType.YA_CHATS;
    }

    @Override
    public YaChatsNotification.Builder toBuilder() {
        return new YaChatsNotification.Builder(this);
    }

    @Nullable
    public String getLogin() {
        return login;
    }

    @Nullable
    public String getGroupId() {
        return groupId;
    }

    public String getTextTemplate() {
        return textTemplate;
    }

    @Override
    public String toString() {
        return "YaChatsNotification{" +
            "textTemplate='" + textTemplate + '\'' +
            ((login == null) ? "" : ", login='" + login + '\'') +
            ((groupId == null) ? "" : ", groupId='" + groupId + '\'') +
            "} " + super.toString();
    }

    @NotThreadSafe
    @ParametersAreNonnullByDefault
    public static class Builder extends Notification.Builder<YaChatsNotification, YaChatsNotification.Builder> {

        private static final String YA_MESSENGER_TG_GROUP_PREFIX = "0/0/tg";

        @Nullable
        private String login = "";
        @Nullable
        private String groupId;
        private String textTemplate;

        public Builder() {
        }

        public Builder(YaChatsNotification notification) {
            super(notification);
            this.login = notification.login;
            this.groupId = notification.groupId;
            this.textTemplate = notification.textTemplate;
        }

        public Builder(TelegramNotification telegramNotification) {
            super(telegramNotification);
            // using default template for YM if default TG template is used
            // for custom templates we have no good options - HTML markdown won't work in YM
            this.textTemplate = telegramNotification.getTextTemplate();

            boolean isGroup = StringUtils.isEmpty(telegramNotification.getLogin());
            if (isGroup) {
                this.groupId = YA_MESSENGER_TG_GROUP_PREFIX + telegramNotification.getChatId();
            } else {
                this.login = telegramNotification.getLogin();
            }
        }

        @Override
        protected YaChatsNotification.Builder self() {
            return this;
        }

        @Override
        public YaChatsNotification build() {
            return new YaChatsNotification(this);
        }

        public YaChatsNotification.Builder setLogin(String login) {
            this.login = login;
            return this;
        }

        public YaChatsNotification.Builder setGroupId(String groupId) {
            this.groupId = groupId;
            return this;
        }

        public YaChatsNotification.Builder setTextTemplate(String textTemplate) {
            this.textTemplate = textTemplate;
            return this;
        }
    }
}
