package ru.yandex.solomon.alert.notification.state;

import java.net.http.HttpConnectTimeoutException;
import java.time.Instant;
import java.util.EnumSet;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.atomic.AtomicReference;

import com.google.common.base.Throwables;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.solomon.alert.notification.ChannelMetrics;
import ru.yandex.solomon.alert.notification.NotificationState;
import ru.yandex.solomon.alert.notification.channel.Event;
import ru.yandex.solomon.alert.notification.channel.NotificationStatus;
import ru.yandex.solomon.alert.notification.domain.NotificationType;

import static ru.yandex.misc.concurrent.CompletableFutures.whenComplete;

/**
 * @author Vladimir Gordiychuk
 */
abstract class AbstractSendState extends AbstractState {
    protected static final Logger logger = LoggerFactory.getLogger(SendingState.class);
    private static final Set<NotificationStatus.Code> SUCCESS_STATUSES = EnumSet.of(
            NotificationStatus.Code.SUCCESS,
            NotificationStatus.Code.SKIP_BY_STATUS,
            NotificationStatus.Code.SKIP_REPEAT);

    protected final EventProcessingTask eventProcessingTask;
    protected final AtomicReference<EventProcessingTask> delayedProcessingTask = new AtomicReference<>();

    AbstractSendState(
            StateContext context,
            NotificationState notificationState,
            EventProcessingTask eventProcessingTask)
    {
        super(context, notificationState);
        this.eventProcessingTask = eventProcessingTask;
    }

    final CompletableFuture<NotificationStatus> sendPreparedMessage() {
        ChannelMetrics metrics = context.getMetrics();
        long startTime = metrics.started(eventProcessingTask.getEvent());
        CompletableFutures.safeCall(() -> context.getChannel().send(getNotificationState().getLatestSuccessNotify(), eventProcessingTask.getEvent()))
                .exceptionally(completionException -> {
                    Throwable error = CompletableFutures.unwrapCompletionException(completionException);
                    logger.warn("Error occurs during send event {} to channel {}",
                            eventProcessingTask.getEvent(), context.getChannel(), error);
                    if (error instanceof HttpConnectTimeoutException) {
                        // May indicate inability to connect to some IPs of the DNS record. HttpClient tries only the first one
                        return NotificationStatus.ERROR_ABLE_TO_RETRY.withDescription(error.getMessage());
                    }
                    return NotificationStatus.ERROR.withDescription(Throwables.getStackTraceAsString(error));
                })
                .thenAcceptAsync(status -> {
                    metrics.completed(eventProcessingTask.getEvent(), status, startTime);
                    if (SUCCESS_STATUSES.contains(status.getCode())) {
                        moveToPending(status);
                        return;
                    }

                    if (context.getChannel().getType() != NotificationType.UNKNOWN) {
                        logger.warn("notification {} failed with status: {}", context.getChannel(), status);
                    }
                    if (status.getCode() == NotificationStatus.Code.ERROR_ABLE_TO_RETRY
                            || status.getCode() == NotificationStatus.Code.RESOURCE_EXHAUSTED) {
                        moveToRetry(status);
                        return;
                    }

                    moveToPending(status);
                }, context.getExecutorService())
                .whenComplete((ignore, ignore2) -> processDelayed());

        return eventProcessingTask.getFuture();
    }

    protected void moveToPending(NotificationStatus prev) {
        moveToPendingWithoutComplete(prev);
        eventProcessingTask.getFuture().complete(prev);
    }

    private void moveToPendingWithoutComplete(NotificationStatus status) {
        NotificationState next = getNotificationState().nextStatus(status, eventProcessingTask.getEvent().getState());
        tryChangeState(new PendingState(context, next));
    }

    protected void moveToRetry(NotificationStatus status) {
        NotificationState next = getNotificationState().nextStatus(status, eventProcessingTask.getEvent().getState());
        RetryingState retry = new RetryingState(context, next, eventProcessingTask, 1, 0);
        if (tryChangeState(retry)) {
            retry.scheduleRetry(status);
        } else {
            eventProcessingTask.getFuture().complete(NotificationStatus.OBSOLETE);
        }
    }

    @Override
    protected boolean isFresherThanNotified(Event event) {
        Instant sending = eventProcessingTask.getEvent().getState().getLatestEval();
        Instant current = event.getState().getLatestEval();
        return current.isAfter(sending);
    }

    protected CompletableFuture<NotificationStatus> delayProcessing(Event event) {
        EventProcessingTask next = new EventProcessingTask(event);
        EventProcessingTask prev;
        do {
            prev = delayedProcessingTask.get();
            if (!isActualTask(prev, next)) {
                return CompletableFuture.completedFuture(NotificationStatus.OBSOLETE
                        .withDescription("Already scheduled send about more actual event"));
            }
        } while (!delayedProcessingTask.compareAndSet(prev, next));

        if (prev != null) {
            prev.getFuture()
                    .complete(NotificationStatus.OBSOLETE
                            .withDescription("More actual event scheduled to send"));
        }

        if (context.getCurrentChannelState() != this) {
            if (delayedProcessingTask.compareAndSet(next, null)) {
                return processByActualState(event);
            }
        }

        return next.getFuture();
    }

    protected void processDelayed() {
        EventProcessingTask delayed = delayedProcessingTask.getAndSet(null);
        if (delayed == null) {
            return;
        }

        CompletableFuture<NotificationStatus> future = delayed.getFuture();
        whenComplete(processByActualState(delayed.getEvent()), future);
    }

    private boolean isActualTask(EventProcessingTask prev, EventProcessingTask next) {
        if (prev == null) {
            return true;
        }

        Instant prevTime = prev.getEvent().getState().getLatestEval();
        Instant nextTime = next.getEvent().getState().getLatestEval();
        return nextTime.isAfter(prevTime);
    }
}
