package ru.yandex.solomon.alert.notification.state;

import java.time.Duration;
import java.time.Instant;
import java.util.concurrent.CompletableFuture;

import javax.annotation.Nullable;

import ru.yandex.solomon.alert.EvaluationStatus;
import ru.yandex.solomon.alert.notification.NotificationState;
import ru.yandex.solomon.alert.notification.channel.Event;
import ru.yandex.solomon.alert.notification.channel.NotificationStatus;

/**
 * @author Vladimir Gordiychuk
 */
public abstract class AbstractState implements NotificationChannelState {
    protected final StateContext context;
    private final NotificationState notificationState;

    AbstractState(StateContext context, NotificationState state) {
        this.context = context;
        this.notificationState = state;
    }

    protected boolean isFresherThanNotified(Event event) {
        Instant latestNotifiedEval = notificationState.getLatestEval();
        Instant latestEval = event.getState().getLatestEval();
        return latestNotifiedEval.isBefore(latestEval);
    }

    private boolean isSubscribedOn(Event event) {
        EvaluationStatus status = event.getState().getStatus();
        return context.getDispatchRule().isSubscribedOn(status.getCode());
    }

    protected boolean isNewEvaluationStatus(Event event) {
        Instant since = event.getState().getSince();
        Instant latestNotify = notificationState.getLatestSuccessNotify();
        return latestNotify.isBefore(since);
    }

    private boolean isTimeToRepeat(Event event) {
        EvaluationStatus.Code code = event.getState().getStatus().getCode();
        Duration past = Duration.between(notificationState.getLatestSuccessNotify(), event.getState().getLatestEval());
        return context.getDispatchRule().isTimeToRepeat(code, past);
    }

    private boolean isMuted(Event event) {
        if (context.getDispatchRule().ignoreMute()) {
            return false;
        }

        return event.getMuteStatus().isMuted();
    }

    protected boolean wasReportedWithSameStatus(Event event) {
        var latestNotifiedEvalStatusCode = getNotificationState().getLatestNotifiedEvalStatusCode();
        return latestNotifiedEvalStatusCode == event.getState().getStatus().getCode();
    }

    boolean tryChangeState(NotificationChannelState next) {
        return context.tryChangeState(this, next);
    }

    CompletableFuture<NotificationStatus> processByActualState(Event event) {
        NotificationChannelState upToDateState = context.getCurrentChannelState();
        if (upToDateState == null) {
            return CompletableFuture.completedFuture(NotificationStatus.OBSOLETE);
        }

        return upToDateState.process(event);
    }

    public NotificationState getNotificationState() {
        return notificationState;
    }

    boolean isSkippedAsFirstOk(Event event) {
        if (!context.getDispatchRule().skipFirstOk()) {
            return false;
        }
        return getNotificationState().getLatestEval().equals(Instant.EPOCH) &&
            event.getState().getStatus().getCode() == EvaluationStatus.Code.OK;
    }

    @Nullable
    protected NotificationStatus ensureNeedsSend(Event event) {
        if (!isFresherThanNotified(event)) {
            return NotificationStatus.OBSOLETE.withDescription("More actual event already was send");
        }

        if (!isSubscribedOn(event) || isSkippedAsFirstOk(event)) {
            return NotificationStatus.SKIP_BY_STATUS;
        }

        if (isMuted(event)) {
            return NotificationStatus.MUTED;
        }

        if (!isNewEvaluationStatus(event) && !isTimeToRepeat(event)) {
            return NotificationStatus.SKIP_REPEAT;
        }

        return null;
    }

    @Override
    public String toString() {
        return "AbstractState{context=" + context + ", notificationState=" + notificationState + '}';
    }
}
