package ru.yandex.solomon.alert.notification.state;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.alert.notification.NotificationState;
import ru.yandex.solomon.alert.notification.RetryOptions;
import ru.yandex.solomon.alert.notification.channel.Event;
import ru.yandex.solomon.alert.notification.channel.NotificationStatus;

import static java.util.concurrent.CompletableFuture.completedFuture;
import static ru.yandex.misc.concurrent.CompletableFutures.whenComplete;

/**
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public class RetryingState extends AbstractSendState {
    private final int attemptNumber;
    private final long spendMillis;
    private final long startTime;
    @Nullable
    private volatile ScheduledFuture<?> retryTask;

    public RetryingState(
            StateContext context,
            NotificationState notificationState,
            EventProcessingTask eventProcessingTask,
            int attemptNumber,
            long spendMillis)
    {
        super(context, notificationState, eventProcessingTask);
        this.attemptNumber = attemptNumber;
        this.spendMillis = spendMillis;
        this.startTime = System.currentTimeMillis();
    }

    void scheduleRetry(NotificationStatus prev) {
        if (retryTask != null) {
            throw new IllegalStateException("Not able start retry task twice:" + eventProcessingTask.getEvent());
        }

        RetryOptions opt = context.getRetryOptions();
        long delayMillis = Math.round(opt.getInitialDelayMillis()
                * Math.pow(opt.getRetryDelayMultiplier(), attemptNumber - 1));

        delayMillis = Math.max(prev.getRetryAfterMillisHint(), delayMillis);
        delayMillis = Math.min(opt.getMaxRetryDelayMillis(), delayMillis);
        delayMillis += ThreadLocalRandom.current().nextLong(opt.getInitialDelayMillis());

        retryTask = context.getExecutorService()
                .schedule(this::runScheduledRetry, delayMillis, TimeUnit.MILLISECONDS);
    }

    private void runScheduledRetry() {
        if (eventProcessingTask.getFuture().isDone()) {
            return;
        }

        EventProcessingTask moreNewTask = delayedProcessingTask.getAndSet(null);
        if (moreNewTask == null) {
            context.getMetrics().retry(eventProcessingTask.getEvent());
            sendPreparedMessage();
            return;
        }

        long nextSpendMillis = System.currentTimeMillis() - startTime + spendMillis;
        NotificationState notificationState = getNotificationState();
        RetryingState retry = new RetryingState(context, notificationState, moreNewTask, attemptNumber, nextSpendMillis);
        eventProcessingTask.getFuture().complete(notificationState.getLatestStatus());
        if (tryChangeState(retry)) {
            retry.runScheduledRetry();
        } else {
            whenComplete(processByActualState(moreNewTask.getEvent()), moreNewTask.getFuture());
        }

        processDelayed();
    }

    @Override
    protected void moveToRetry(NotificationStatus status) {
        RetryOptions opts = context.getRetryOptions();
        int nextAttempt = attemptNumber + 1;
        long nextSpendMillis = System.currentTimeMillis() - startTime + spendMillis;
        if (nextAttempt >= opts.getTaskRetryLimit() || isRetryLimitReach(nextSpendMillis)) {
            moveToPending(status);
            return;
        }

        NotificationState next = getNotificationState().nextStatus(status, eventProcessingTask.getEvent().getState());

        final EventProcessingTask actualTask;
        EventProcessingTask delayed = delayedProcessingTask.getAndSet(null);
        if (delayed != null) {
            eventProcessingTask.getFuture().complete(next.getLatestStatus());
            actualTask = delayed;
        } else {
            actualTask = eventProcessingTask;
        }

        RetryingState retry = new RetryingState(context, next, actualTask, nextAttempt, nextSpendMillis);
        if (tryChangeState(retry)) {
            retry.scheduleRetry(status);
        } else {
            actualTask.getFuture().complete(NotificationStatus.OBSOLETE);
        }
    }

    private boolean isRetryLimitReach(long spendMillis) {
        return spendMillis >= context.getRetryOptions().getTaskAgeLimitMillis();
    }

    @Override
    public CompletableFuture<NotificationStatus> process(Event event) {
        NotificationStatus status = ensureNeedsSend(event);
        if (status != null) {
            return completedFuture(status);
        }

        return delayProcessing(event);
    }

    @Override
    public void cancel() {
        moveToPending(NotificationStatus.OBSOLETE);
        ScheduledFuture<?> retryTask = this.retryTask;
        if (retryTask != null) {
            retryTask.cancel(true);
        }
    }
}
