package ru.yandex.solomon.alert.notification.state;

import java.time.Instant;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.atomic.AtomicReferenceFieldUpdater;

import javax.annotation.ParametersAreNonnullByDefault;
import javax.annotation.WillNotClose;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.solomon.alert.api.converters.AlertConverter;
import ru.yandex.solomon.alert.api.converters.NotificationConverter;
import ru.yandex.solomon.alert.domain.Alert;
import ru.yandex.solomon.alert.domain.ChannelConfig;
import ru.yandex.solomon.alert.notification.ChannelMetrics;
import ru.yandex.solomon.alert.notification.DispatchRule;
import ru.yandex.solomon.alert.notification.NotificationKey;
import ru.yandex.solomon.alert.notification.NotificationServiceMetrics;
import ru.yandex.solomon.alert.notification.NotificationState;
import ru.yandex.solomon.alert.notification.RetryOptions;
import ru.yandex.solomon.alert.notification.channel.Event;
import ru.yandex.solomon.alert.notification.channel.NotificationChannel;
import ru.yandex.solomon.alert.notification.channel.NotificationStatus;
import ru.yandex.solomon.alert.protobuf.TPersistNotificationState;
import ru.yandex.solomon.alert.rule.AlertProcessingState;

/**
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public class StatefulNotificationChannelImpl implements StatefulNotificationChannel, StateContext, AutoCloseable {
    private static final AtomicReferenceFieldUpdater<StatefulNotificationChannelImpl, NotificationChannelState> STATE_FILED =
        AtomicReferenceFieldUpdater.newUpdater(StatefulNotificationChannelImpl.class, NotificationChannelState.class, "state");

    private static final Logger logger = LoggerFactory.getLogger(StatefulNotificationChannelImpl.class);

    @WillNotClose
    private final ScheduledExecutorService executorService;
    private final Alert alert;
    private final ChannelConfig channelConfigOverride;
    private final NotificationChannel channel;
    private final RetryOptions retryOptions;
    private final NotificationServiceMetrics metrics;
    private final NotificationConverter notificationConverter;
    private volatile NotificationChannelState state;

    public StatefulNotificationChannelImpl(
        Alert alert,
        NotificationChannel channel,
        ChannelConfig channelConfigOverride,
        ScheduledExecutorService executorService,
        RetryOptions retryOptions,
        NotificationServiceMetrics metrics,
        NotificationConverter notificationConverter)
    {
        this.executorService = executorService;
        this.alert = alert;
        this.channel = channel;
        this.retryOptions = retryOptions;
        this.metrics = metrics;
        this.notificationConverter = notificationConverter;
        this.channelConfigOverride = channelConfigOverride;
        this.state = new PendingState(this, NotificationState.init(NotificationKey.of(alert, channel.getId())));
    }

    // For testing purposes only
    public void overrideState(AbstractState newState) {
        this.state = newState;
    }

    public CompletableFuture<NotificationStatus> send(AlertProcessingState evaluationState) {
        return this.state.process(new Event(alert, evaluationState));
    }

    @Override
    public void close() {
        state.cancel();
    }

    @Override
    public NotificationChannel getChannel() {
        return channel;
    }

    @Override
    public ScheduledExecutorService getExecutorService() {
        return executorService;
    }

    @Override
    public RetryOptions getRetryOptions() {
        return retryOptions;
    }

    @Override
    public ChannelMetrics getMetrics() {
        return metrics.getChannelMetrics(channel.getType());
    }

    @Override
    public boolean tryChangeState(NotificationChannelState current, NotificationChannelState next) {
        if (STATE_FILED.compareAndSet(this, current, next)) {
            if (logger.isDebugEnabled()) {
                NotificationKey key = new NotificationKey(alert.getKey(), channel.getId());
                logger.debug("Move state for key {} from {} to {}", key, current.getClass().getSimpleName(), next.getClass().getSimpleName());
            }
            return true;
        } else {
            return false;
        }
    }

    @Override
    public NotificationChannelState getCurrentChannelState() {
        return state;
    }

    @Override
    public DispatchRule getDispatchRule() {
        return channel.getDispatchRule(channelConfigOverride);
    }

    public NotificationState getLatestNotificationState() {
        return state.getNotificationState();
    }

    public TPersistNotificationState dumpState() {
        NotificationState state = this.state.getNotificationState();
        var builder = TPersistNotificationState.newBuilder()
            .setNotificationChannelId(channel.getId())
            .setLatestEvalMillis(state.getLatestEval().toEpochMilli())
            .setLatestSuccessMillis(state.getLatestSuccessNotify().toEpochMilli())
            .setStatus(notificationConverter.statusToProto(state.getLatestStatus()));

        var evalCode = state.getLatestNotifiedEvalStatusCode();
        if (evalCode != null) {
            builder.setLatestNotifiedEvalStatusCode(AlertConverter.statusCodeToProto(evalCode));
        }

        return builder.build();
    }

    public void restoreState(TPersistNotificationState proto) {
        if (proto.getLatestEvalMillis() == 0) {
            return;
        }

        NotificationState state = NotificationState.newBuilder()
            .setKey(NotificationKey.of(alert, channel.getId()))
            .setLatestEval(Instant.ofEpochMilli(proto.getLatestEvalMillis()))
            .setLatestSuccessNotify(Instant.ofEpochMilli(proto.getLatestSuccessMillis()))
            .setLatestStatus(notificationConverter.protoToStatus(proto.getStatus()))
            .setLatestNotifiedEvalStatusCode(AlertConverter.protoToStatusCode(proto.getLatestNotifiedEvalStatusCode()))
            .build();

        NotificationChannelState curr = this.state;
        NotificationChannelState next = new PendingState(this, state);
        if (tryChangeState(curr, next)) {
            curr.cancel();
        }
    }

    @Override
    public boolean isDefault() {
        return channel.isDefault();
    }

    @Override
    public String toString() {
        return "StatefulNotificationChannelImpl{" +
            " alert=" + alert.getKey() +
            ", channel=" + channel.getId() +
            ", state=" + state.getClass() +
            '}';
    }
}
