package ru.yandex.solomon.alert.rule;

import java.time.Duration;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Function;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.monlib.metrics.meter.ExpMovingAverage;
import ru.yandex.monlib.metrics.meter.Meter;
import ru.yandex.solomon.alert.EvaluationStatus;
import ru.yandex.solomon.alert.domain.Alert;
import ru.yandex.solomon.alert.rule.usage.AlertRuleMetrics;
import ru.yandex.solomon.alert.util.ExceptionResolver;
import ru.yandex.solomon.alert.util.ResovledByUUIDRuntimeException;
import ru.yandex.solomon.expression.analytics.GraphDataLoadRequest;
import ru.yandex.solomon.flags.FeatureFlag;
import ru.yandex.solomon.flags.FeatureFlagsHolder;
import ru.yandex.solomon.labels.LabelKeys;
import ru.yandex.solomon.labels.query.Selector;
import ru.yandex.solomon.labels.query.Selectors;
import ru.yandex.solomon.math.operation.Metric;
import ru.yandex.solomon.metrics.client.MetabaseStatus;
import ru.yandex.solomon.metrics.client.MetricsClient;
import ru.yandex.solomon.metrics.client.StockpileStatus;
import ru.yandex.solomon.metrics.client.combined.FindAndReadManyRequest;
import ru.yandex.solomon.metrics.client.combined.FindAndReadManyResponse;
import ru.yandex.solomon.metrics.client.exceptions.TooManyMetricsLoadedBySelectors;
import ru.yandex.solomon.model.MetricKey;
import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.solomon.model.timeseries.AggrGraphDataIterable;
import ru.yandex.stockpile.api.EStockpileStatusCode;

import static java.util.stream.Collectors.collectingAndThen;
import static java.util.stream.Collectors.toList;
import static ru.yandex.solomon.alert.rule.EvaluationStatusMapping.toEvaluationStatus;
import static ru.yandex.solomon.alert.rule.ResultOrProceed.proceed;
import static ru.yandex.solomon.alert.rule.ResultOrProceed.ready;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public abstract class AbstractAlertRule implements AlertRule {

    private static final Logger logger = LoggerFactory.getLogger(AbstractAlertRule.class);

    protected final Alert alert;

    protected final AlertRuleMetrics metrics;
    private final MetricsClient cachingMetricsClient;

    protected final Duration period;

    protected final Meter ioTime = Meter.of(ExpMovingAverage.fifteenMinutes());
    protected final Meter cpuTime = Meter.of(ExpMovingAverage.fifteenMinutes());

    protected final FeatureFlagsHolder featureFlags;

    protected AbstractAlertRule(Alert alert, MetricsClient cachingMetricsClient, AlertRuleMetrics metrics, FeatureFlagsHolder featureFlags) {
        this.alert = alert;
        this.period = alert.getPeriod();
        this.cachingMetricsClient = cachingMetricsClient;
        this.metrics = metrics;
        this.featureFlags = featureFlags;
    }

    @Nonnull
    @Override
    public String getId() {
        return alert.getId();
    }

    @Nonnull
    @Override
    public Alert getAlert() {
        return alert;
    }

    @Override
    public Meter getIoTime() {
        return ioTime;
    }

    @Override
    public Meter getCpuTime() {
        return cpuTime;
    }

    protected EvaluationStatus tooManyMetrics() {
        return EvaluationStatus.ERROR.withDescription("Max metrics to load limit exceeded: "
                + alert.getMetricsLimit()
                + ". Try to specify one more group label.");
    }

    @Nullable
    protected abstract EvaluationStatus applyNoMetricsPolicy(Function<EvaluationStatus.Code, EvaluationStatus> statusMapper);

    @Nullable
    protected abstract EvaluationStatus applyNoPointsPolicy(Function<EvaluationStatus.Code, EvaluationStatus> statusMapper);

    protected ResultOrProceed<EvaluationStatus, List<Metric<MetricKey>>> checkReadManyResponse(
            GraphDataLoadRequest loadRequest,
            FindAndReadManyResponse response,
            @Nullable Throwable throwable)
    {
        this.metrics.readRequests.inc();
        if (throwable != null) {
            Throwable ex = CompletableFutures.unwrapCompletionException(throwable);
            if (ex instanceof TooManyMetricsLoadedBySelectors) {
                return ready(tooManyMetrics());
            }
            if (ex instanceof RuntimeException) {
                throw (RuntimeException) ex;
            }
            throw new RuntimeException(ex);
        }
        MetabaseStatus metaStatus = response.getMetaStatus();
        StockpileStatus storageStatus = response.getStorageStatus();
        List<Metric<MetricKey>> metrics = response.getMetrics();
        int points = 0;
        for (var metric : metrics) {
            if (metric.getTimeseries() != null) {
                points += metric.getTimeseries().getRecordCount();
            }
        }
        this.metrics.readMetrics.add(metrics.size());
        this.metrics.readPoints.add(points);

        if (!metaStatus.isOkOrNoData()) {
            EvaluationStatus status = toEvaluationStatus(metaStatus, "on find metrics " + loadRequest.getSelectors());
            this.metrics.recordReadFailure(status);
            return ready(status);
        }

        if (storageStatus.getCode() != EStockpileStatusCode.OK) {
            EvaluationStatus status = toEvaluationStatus(storageStatus, "on read " + loadRequest.getSelectors());
            this.metrics.recordReadFailure(status);
            logger.warn("{} readMany fail cause by {}:{}", alert.getKey(), status.getCode(), status.getDescription());
            return ready(status);
        }

        if (metrics.isEmpty()) {
            @Nullable EvaluationStatus status = applyNoMetricsPolicy(ruleCode -> {
                if (metaStatus.isOk()) {
                    logger.debug("{} - for alert {} caused by no metrics by selector {}",
                            ruleCode, getId(), loadRequest.getSelectors());
                } else {
                    logger.warn("{} - for alert {} caused by no metrics by selector {} ({})",
                            ruleCode, getId(), loadRequest.getSelectors(), metaStatus);
                }
                return ruleCode.toStatus("No metrics found by selectors: " + loadRequest.getSelectors());
            });
            if (status != null) {
                return ready(status);
            }
        }

        if (metrics.size() > alert.getMetricsLimit()) { // probably dead code unless findAndReadMany ignores limit parameter
            EvaluationStatus status = tooManyMetrics();
            this.metrics.recordReadFailure(status);
            return ready(status);
        }

        return proceed(metrics);
    }

    protected static EvaluationStatus mergeStatuses(/* @NonEmpty */ List<EvaluationStatus> statuses) {
        // Merging is probably an overkill since all statuses are generated by the same policy
        EvaluationStatus merged = statuses.get(0);
        for (EvaluationStatus status : statuses) {
            // Codes are ordered in an increasing severity order
            if (status.getCode().ordinal() > merged.getCode().ordinal()) {
                merged = status;
            }
        }
        return merged;
    }

    public record LoadRequestAndResult(GraphDataLoadRequest origRequest, List<Metric<MetricKey>> metrics) {

    }
    public ResultOrProceed<EvaluationStatus, MultipleTimeSeries> convertToManyTimeSeries(LoadRequestAndResult loadRequestAndResult) {
        return loadRequestAndResult.metrics().stream()
                .map(metric -> makeSingleTimeSeries(metric.getKey(), metric.getType(), metric.getTimeseries()))
                .collect(collectingAndThen(toList(), ResultOrProceed.allOf(AbstractAlertRule::mergeStatuses)))
                .map(timeSeriesList -> new MultipleTimeSeries(loadRequestAndResult.origRequest(), timeSeriesList));
    }

    private ResultOrProceed<EvaluationStatus, SingleTimeSeries> makeSingleTimeSeries(
            MetricKey metricKey,
            MetricType type,
            AggrGraphDataIterable source)
    {
        if (source.isEmpty()) {
            @Nullable EvaluationStatus status = applyNoPointsPolicy(code -> {
                return code.toStatus("No points in metric " + metricKey.getLabels());
            });
            if (status != null) {
                return ready(status);
            }
        }

        return proceed(new SingleTimeSeries(metricKey, type, source));
    }

    protected EvaluationStatus classifyError(String alertId, Throwable e) {
        Throwable cause = CompletableFutures.unwrapCompletionException(e);
        RuntimeException exception = ExceptionResolver.resolve(e);
        if (exception instanceof ResovledByUUIDRuntimeException uuidRuntimeException) {
            logger.error("Alert {} in ERROR caused by unhandled exception {}", alertId, uuidRuntimeException.getUuid(), cause);
        } else {
            logger.debug("Alert {} in ERROR caused by", alertId, cause);
        }
        return EvaluationStatus.ERROR.withDescription(exception.getMessage());
    }

    protected boolean throughDataProxy(Selectors selectors) {
        Selector projectSelector = selectors.findByKey(LabelKeys.PROJECT);
        if (projectSelector == null || !projectSelector.isExact()) {
            return false;
        }
        if (!featureFlags.hasFlag(FeatureFlag.ALERTING_DATAPROXY, projectSelector.getValue())) {
            return false;
        }

        if (alert.isObtainedFromTemplate() && isLegacyYasmQuery(selectors)) {
            return false;
        }

        return true;
    }

    private static boolean isLegacyYasmQuery(Selectors selectors) {
        @Nullable Selector maybeCluster = selectors.findByKey(LabelKeys.CLUSTER);
        @Nullable Selector maybeService = selectors.findByKey(LabelKeys.SERVICE);

        return maybeCluster != null || maybeService != null;
    }

    protected CompletableFuture<FindAndReadManyResponse> findAndReadMany(FindAndReadManyRequest.Builder request) {
        if (alert.isObtainedFromTemplate() && isLegacyYasmQuery(request.getSelectors())) {
            request.forbidDataProxy();
        }
        return cachingMetricsClient.findAndReadMany(request.build());
    }
}
