package ru.yandex.solomon.alert.rule;

import java.time.Duration;
import java.time.Instant;
import java.util.concurrent.CompletableFuture;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableList;

import ru.yandex.monlib.metrics.meter.ExpMovingAverage;
import ru.yandex.monlib.metrics.meter.Meter;
import ru.yandex.solomon.alert.EvaluationStatus;
import ru.yandex.solomon.alert.domain.Alert;
import ru.yandex.solomon.alert.evaluation.AlertRuleExecutorType;

/**
 * Prepare to execute alert rule
 *
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public interface AlertRule {
    /**
     * @see Alert#getId()
     */
    @Nonnull
    String getId();

    @Nonnull
    Alert getAlert();

    /**
     * Evaluate rule on specified time, this time can be processed as end for time window
     */
    CompletableFuture<EvaluationStatus> eval(Instant now, AlertRuleDeadlines deadlines);

    /**
     * Evaluate rule and provide time series used to define evaluation status
     */
    default CompletableFuture<ExplainResult> explain(Instant now, AlertRuleDeadlines deadlines) {
        return eval(now, deadlines)
                .thenApply(status -> new ExplainResult(status, ImmutableList.of()));
    }

    default Meter getIoTime() {
        return Meter.of(ExpMovingAverage.fifteenMinutes());
    }

    default Meter getCpuTime() {
        return Meter.of(ExpMovingAverage.fifteenMinutes());
    }

    CompletableFuture<SimulationResult> simulate(Instant from, Instant to, Duration gridStep, AlertRuleDeadlines deadlines);

    default AlertRuleExecutorType getExecutorType() {
        return AlertRuleExecutorType.DEFAULT;
    }
}
