package ru.yandex.solomon.alert.rule;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.alert.domain.Alert;
import ru.yandex.solomon.alert.domain.SubAlert;
import ru.yandex.solomon.alert.domain.expression.ExpressionAlert;
import ru.yandex.solomon.alert.domain.threshold.ThresholdAlert;
import ru.yandex.solomon.alert.rule.expression.ExpressionAlertRule;
import ru.yandex.solomon.alert.rule.expression.ExpressionAlertTemplateProcessor;
import ru.yandex.solomon.alert.rule.threshold.ThresholdAlertRule;
import ru.yandex.solomon.alert.rule.threshold.ThresholdAlertTemplateProcessor;
import ru.yandex.solomon.alert.rule.usage.AlertRuleMetrics;
import ru.yandex.solomon.alert.rule.usage.ProjectAlertRuleMetrics;
import ru.yandex.solomon.alert.statuses.AlertingStatusesSelector;
import ru.yandex.solomon.alert.template.TemplateFactory;
import ru.yandex.solomon.flags.FeatureFlagsHolder;
import ru.yandex.solomon.metrics.client.MetricsClient;

/**
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public class AlertRuleFactoryImpl implements AlertRuleFactory {
    private static final Logger logger = LoggerFactory.getLogger(AlertRuleFactoryImpl.class);

    private final ProgramCompiler compiler;
    private final ProjectAlertRuleMetrics metrics;
    private final MetricsClient cachingMetricsClient;
    private final TemplateFactory templateFactory;
    private final AlertingStatusesSelector alertingStatuses;
    private final FeatureFlagsHolder featureFlags;

    public AlertRuleFactoryImpl(
            MetricsClient cachingMetricsClient,
            ProjectAlertRuleMetrics metrics,
            TemplateFactory templateFactory,
            @Nullable AlertingStatusesSelector alertingStatuses,
            FeatureFlagsHolder featureFlags)
    {
        this.compiler = new ProgramCompiler();
        this.cachingMetricsClient = cachingMetricsClient;
        this.metrics = metrics;
        this.templateFactory = templateFactory;
        this.alertingStatuses = alertingStatuses;
        this.featureFlags = featureFlags;
    }

    @Nonnull
    @Override
    public AlertRule createAlertRule(Alert alert) {
        try {
            var metrics = this.metrics.of(alert.getProjectId());
            switch (alert.getAlertType()) {
                case SUB_ALERT:
                    return createSubAlertRule((SubAlert) alert, metrics);
                case THRESHOLD: {
                    ThresholdAlert thresholdAlert = (ThresholdAlert) alert;
                    return new ThresholdAlertRule(
                            thresholdAlert,
                            cachingMetricsClient,
                            metrics,
                            new ThresholdAlertTemplateProcessor(thresholdAlert, templateFactory),
                            compiler.compile(thresholdAlert),
                            featureFlags);
                }
                case EXPRESSION: {
                    ExpressionAlert expressionAlert = (ExpressionAlert) alert;
                    return new ExpressionAlertRule(
                            expressionAlert,
                            cachingMetricsClient,
                            metrics,
                            new ExpressionAlertTemplateProcessor(expressionAlert, templateFactory, Labels.of()),
                            compiler.compile(expressionAlert),
                            alertingStatuses,
                            featureFlags);
                }
                default:
                    throw new UnsupportedOperationException("Unsupported type for alert: " + alert);
            }
        } catch (Throwable e) {
            logger.error("Failed create alert rule by alert {}, ", alert, e);
            return new FailedInitAlertRule(alert, e);
        }
    }

    private AlertRule createSubAlertRule(SubAlert subAlert, AlertRuleMetrics metrics) {
        switch (subAlert.getParent().getAlertType()) {
            case THRESHOLD -> {
                ThresholdAlert parent = (ThresholdAlert) subAlert.getParent();
                return new ThresholdAlertRule(
                        subAlert,
                        cachingMetricsClient,
                        metrics,
                        new ThresholdAlertTemplateProcessor(parent, templateFactory),
                        compiler.compile(parent),
                        featureFlags);
            }
            case EXPRESSION -> {
                ExpressionAlert parent = (ExpressionAlert) subAlert.getParent();
                return new ExpressionAlertRule(
                        subAlert,
                        cachingMetricsClient,
                        metrics,
                        new ExpressionAlertTemplateProcessor(parent, templateFactory, subAlert.getGroupKey()),
                        compiler.compile(parent),
                        alertingStatuses,
                        featureFlags);
            }
            default -> throw new UnsupportedOperationException("Unsupported type for multi alert: " + subAlert);
        }
    }
}
