package ru.yandex.solomon.alert.rule;

import java.time.Instant;
import java.util.concurrent.TimeUnit;

import javax.annotation.ParametersAreNonnullByDefault;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public class AlertRuleFairDeadlines implements AlertRuleDeadlines {
    private final long softResolveDeadline;
    private final long softReadDeadline;
    private final long hardDeadline;

    private AlertRuleFairDeadlines(long now, long lag, long alertRuleTimeout) {
        hardDeadline = now + alertRuleTimeout;

        // For 1 minute periodic rule, maxSoftWait = 22s, minSoftWait = 3s
        long maxSoftWait = (3L * alertRuleTimeout) / 4L;
        long minSoftWait = alertRuleTimeout / 10L;

        long softWait;
        double lagInTimeouts = (double) lag / alertRuleTimeout;
        if (lagInTimeouts < 1d) {
            softWait = maxSoftWait;
        } else if (lagInTimeouts > 3d) {
            softWait = minSoftWait;
        } else {
            double fraction = (lagInTimeouts - 1d) / 2d;
            // fraction = 0 -> lagByTimeout = 1 -> maxWait
            // fraction = 1 -> lagByTimeout = 3 -> minWait
            softWait = maxSoftWait - (long)(fraction * (maxSoftWait - minSoftWait));
        }
        softResolveDeadline = softReadDeadline = now + softWait;
    }

    public static AlertRuleDeadlines adjustToLag(long now, long lag, long alertRuleTimeout) {
        return new AlertRuleFairDeadlines(now, lag, alertRuleTimeout);
    }

    public static AlertRuleDeadlines ignoreLag(Instant now, long duration, TimeUnit unit) {
        return new AlertRuleFairDeadlines(now.toEpochMilli(), 0, unit.toMillis(duration));
    }

    @Override
    public long softResolveDeadline() {
        return softResolveDeadline;
    }

    @Override
    public long softReadDeadline() {
        return softReadDeadline;
    }

    @Override
    public long hardDeadline() {
        return hardDeadline;
    }

    @Override
    public String toString() {
        return "AlertRuleFairDeadlines{" +
                "softResolveDeadline=" + softResolveDeadline +
                ", softReadDeadline=" + softReadDeadline +
                ", hardDeadline=" + hardDeadline +
                '}';
    }
}
