package ru.yandex.solomon.alert.rule;

import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.Objects;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.MoreObjects;

import ru.yandex.solomon.alert.EvaluationStatus;
import ru.yandex.solomon.alert.domain.Alert;
import ru.yandex.solomon.alert.domain.AlertKey;

/**
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public class EvaluationState {
    private final AlertKey key;
    private final int alertVersion;
    private final EvaluationStatus status;
    private final Instant since;
    private final Instant latestEval;
    private final EvaluationStatus previousStatus;

    private EvaluationState(Builder builder) {
        this.key = Objects.requireNonNull(builder.key);
        this.alertVersion = builder.alertVersion;
        this.status = Objects.requireNonNull(builder.status);
        this.since = Objects.requireNonNull(builder.since);
        this.latestEval = builder.latestEval == null
                ? this.since
                : builder.latestEval;
        this.previousStatus = builder.previousStatus == null
                ? this.status
                : builder.previousStatus;
    }

    public static Builder newBuilder() {
        return new Builder();
    }

    public static Builder newBuilder(Alert alert) {
        return new Builder()
                .setAlertKey(alert.getKey())
                .setAlertVersion(alert.getVersion());
    }

    public Builder toBuilder() {
        return new Builder(this);
    }

    public EvaluationState nextStatus(EvaluationStatus status, Instant time) {
        var builder = toBuilder()
                .setStatus(status)
                .setLatestEval(time)
                .setPreviousStatus(this.status);

        if (this.status.getCode() != status.getCode()) {
            builder.setSince(time);
        }

        return builder.build();
    }

    public String getAlertId() {
        return key.getAlertId();
    }

    public AlertKey getKey() {
         return key;
    }

    public String getProjectId() {
        return key.getProjectId();
    }

    public int getAlertVersion() {
        return alertVersion;
    }

    public EvaluationStatus getStatus() {
        return status;
    }

    public Instant getSince() {
        return since;
    }

    public Instant getLatestEval() {
        return latestEval;
    }

    public Instant getLatestEvalTruncated() {
        // TODO: truncating should base on evaluation interval (gordiychuk@)
        return latestEval.truncatedTo(ChronoUnit.MINUTES);
    }

    public EvaluationStatus getPreviousStatus() {
        return previousStatus;
    }

    @Override
    public String toString() {
        return MoreObjects.toStringHelper(this)
                .add("alertKey", key)
                .add("alertVersion", alertVersion)
                .add("status", status)
                .add("since", since)
                .add("latestEval", latestEval)
                .add("previousStatus", previousStatus)
                .toString();
    }

    @ParametersAreNonnullByDefault
    public static class Builder {
        private AlertKey key;
        private int alertVersion;
        @Nullable
        private EvaluationStatus status;
        @Nullable
        private Instant since;
        @Nullable
        private Instant latestEval;
        private EvaluationStatus previousStatus = EvaluationStatus.OK;

        private Builder() {
        }

        private Builder(EvaluationState state) {
            this.key = state.key;
            this.alertVersion = state.alertVersion;
            this.status = state.status;
            this.since = state.since;
            this.latestEval = state.latestEval;
            this.previousStatus = state.previousStatus;
        }

        public Builder setAlertKey(AlertKey key) {
            this.key = key;
            return this;
        }

        public Builder setAlertVersion(int alertVersion) {
            this.alertVersion = alertVersion;
            return this;
        }

        public Builder setStatus(EvaluationStatus status) {
            this.status = status;
            return this;
        }

        public Builder setSince(Instant time) {
            this.since = time;
            return this;
        }

        public Builder setLatestEval(Instant time) {
            this.latestEval = time;
            return this;
        }

        public Builder setPreviousStatus(EvaluationStatus status) {
            this.previousStatus = status;
            return this;
        }

        public EvaluationState build() {
            return new EvaluationState(this);
        }
    }
}
