package ru.yandex.solomon.alert.rule;

import java.util.concurrent.atomic.AtomicLong;

import javax.annotation.ParametersAreNonnullByDefault;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public class EwmaAwaitLimiter implements AwaitLimiter {
    private final int decayFactor;
    private final AtomicLong expWeightedAvgElapsedNanos = new AtomicLong(0);
    private final long maxAvgElapsedNanos;

    public EwmaAwaitLimiter(int decayFactor, long maxAvgElapsedNanos) {
        this.decayFactor = decayFactor;
        this.maxAvgElapsedNanos = maxAvgElapsedNanos;
    }

    @Override
    public void update(long elapsedNanos) {
        expWeightedAvgElapsedNanos.getAndUpdate(oldEwma -> oldEwma + (elapsedNanos - oldEwma) / decayFactor);
    }

    @Override
    public long availableWaitNanos() {
        /*
        * Return v such that after updating EMWA with v the EMWA will be equal to maxAvgElapsedNanos
        *
        * e + (v - e) / decayFactor = maxAvgElapsedNanos
        * v = e + (maxAvgElapsedNanos - e) * decayFactor
        *
        * When e is small (i.e. << maxAvgElapsedNanos) v is almost as big as maxAvgElapsedNanos * decayFactor
        * When e is near maxAvgElapsedNanos then v is approximately maxAvgElapsedNanos
        */
        long e = expWeightedAvgElapsedNanos.get();
        return Math.max(e + (maxAvgElapsedNanos - e) * decayFactor, 0L);
    }
}
