package ru.yandex.solomon.alert.rule;

import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;

import ru.yandex.solomon.alert.domain.AbstractAlert;
import ru.yandex.solomon.alert.domain.expression.ExpressionAlert;
import ru.yandex.solomon.alert.domain.threshold.ThresholdAlert;
import ru.yandex.solomon.expression.analytics.Program;
import ru.yandex.solomon.expression.compile.DeprOpts;
import ru.yandex.solomon.expression.version.SelVersion;

/**
 * @author Vladimir Gordiychuk
 */
public class ProgramCompiler {
    public static final SelVersion ALERTING_SEL_VERSION = SelVersion.GROUP_LINES_RETURN_VECTOR_2;
    private static final Program EMPTY = Program.fromSource(ALERTING_SEL_VERSION, "")
            .withDeprOpts(DeprOpts.ALERTING)
            .compile();

    private final Cache<Source, Program> compiledBySource;

    public ProgramCompiler() {
        compiledBySource = CacheBuilder.newBuilder()
                .weakValues()
                .build();
    }

    public Program compile(AbstractAlert alert) {
        var source = Source.of(alert);
        var compiled = compiledBySource.getIfPresent(source);
        if (compiled != null) {
            return compiled;
        }

        compiled = source.compile();
        var prev = compiledBySource.asMap().putIfAbsent(source, compiled);
        if (prev == null) {
            return compiled;
        }
        return prev;
    }

    private interface Source {
        Program compile();

        static Source of(AbstractAlert alert) {
            if (alert instanceof ExpressionAlert expression) {
                return new Expression(expression.getProgram(), expression.getCheckExpression());
            }

            if (alert instanceof ThresholdAlert threshold) {
                return new Transformation(threshold.getTransformations());
            }

            throw new IllegalArgumentException("Unknown alert type");
        }
    }

    private record Expression(String program, String check) implements Source {
        @Override
        public Program compile() {
            var src = ExpressionAlert.combineProgramWithCheck(program, check);
            if (src.isEmpty()) {
                return EMPTY;
            }

            return Program.fromSource(ALERTING_SEL_VERSION, src)
                    .withDeprOpts(DeprOpts.ALERTING)
                    .compile();
        }
    }

    private record Transformation(String transformation) implements Source {
        @Override
        public Program compile() {
            if (transformation.isEmpty()) {
                return EMPTY;
            }

            return Program.fromSource(ALERTING_SEL_VERSION, "")
                    .withExternalExpression(transformation)
                    .withDeprOpts(DeprOpts.ALERTING)
                    .compile();
        }
    }
}
