package ru.yandex.solomon.alert.rule;

import java.util.ArrayList;
import java.util.List;
import java.util.NoSuchElementException;
import java.util.function.Function;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public final class ResultOrProceed<Result, Proceed> {
    final private @Nullable Result result;
    final private @Nullable Proceed proceed;

    public ResultOrProceed(@Nullable Result result, @Nullable Proceed proceed) {
        this.result = result;
        this.proceed = proceed;
    }

    public static <Result, Cont> ResultOrProceed<Result, Cont> ready(Result result) {
        return new ResultOrProceed<>(result, null);
    }

    public static <Result, Proceed> ResultOrProceed<Result, Proceed> proceed(Proceed proceed) {
        return new ResultOrProceed<>(null, proceed);
    }

    public boolean hasResult() {
        return result != null;
    }

    @Nonnull
    public Result terminate(Function<Proceed, Result> terminator) {
        if (hasResult()) {
            return result;
        }
        return terminator.apply(proceed);
    }

    public static <Result, Proceed> Function<List<ResultOrProceed<Result, Proceed>>, ResultOrProceed<Result, List<Proceed>>> allOf(
            Function<List<Result>, Result> mergeResults)
    {
        return x -> allOf(x, mergeResults);
    }

    public static <Result, Proceed> ResultOrProceed<Result, List<Proceed>> allOf(
            List<ResultOrProceed<Result, Proceed>> tries,
            Function<List<Result>, Result> mergeResults)
    {
        List<Proceed> proceeds = new ArrayList<>(tries.size());
        List<Result> results = new ArrayList<>(tries.size());
        for (var t : tries) {
            if (t.hasResult()) {
                results.add(t.getResult());
            } else {
                proceeds.add(t.getIncomplete());
            }
        }
        if (results.isEmpty()) {
            return proceed(proceeds);
        }
        return ready(mergeResults.apply(results));
    }

    public <NextProceed> ResultOrProceed<Result, NextProceed> map(Function<Proceed, NextProceed> terminator) {
        if (hasResult()) {
            return ready(getResult());
        }
        return proceed(terminator.apply(proceed));
    }

    public <NextProceed> ResultOrProceed<Result, NextProceed> compose(Function<Proceed, ResultOrProceed<Result, NextProceed>> other) {
        if (hasResult()) {
            return ready(getResult());
        }
        return other.apply(proceed);
    }

    @Nonnull
    public Result getResult() {
        return terminate((proceed) -> {
            throw new NoSuchElementException("No result present");
        });
    }

    @Nonnull
    public Proceed getIncomplete() {
        if (hasResult()) {
            throw new NoSuchElementException("Already has result");
        }
        return proceed;
    }
}
