package ru.yandex.solomon.alert.rule;

import java.util.List;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Strings;

import ru.yandex.solomon.alert.EvaluationStatus;
import ru.yandex.solomon.alert.protobuf.ERequestStatusCode;
import ru.yandex.solomon.alert.protobuf.TAlertDataSeries;
import ru.yandex.solomon.alert.protobuf.TEvaluationStatus;
import ru.yandex.solomon.alert.protobuf.TSimulateEvaluationResponse;
import ru.yandex.solomon.labels.protobuf.LabelConverter;
import ru.yandex.solomon.metrics.client.StockpileStatus;
import ru.yandex.solomon.model.point.RecyclableAggrPoint;
import ru.yandex.solomon.model.timeseries.AggrGraphDataIterable;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public class SimulationResult {
    private final SimulationStatus code;
    private final String message;

    private final List<AlertTimeSeries> resultingLines;
    private final List<EvaluationStatus.Code> statuses;

    public SimulationResult(SimulationStatus code, String message, List<AlertTimeSeries> resultingLines, List<EvaluationStatus.Code> statuses) {
        this.code = code;
        this.message = message;
        this.resultingLines = resultingLines;
        this.statuses = statuses;
    }

    public SimulationResult(SimulationStatus code, String message) {
        this(code, message, List.of(), List.of());
    }

    public SimulationStatus getCode() {
        return code;
    }

    public String getMessage() {
        return message;
    }

    public List<AlertTimeSeries> getResultingLines() {
        return resultingLines;
    }

    public List<EvaluationStatus.Code> getStatuses() {
        return statuses;
    }

    public static SimulationResult fromStockpile(StockpileStatus status, @Nullable String additional) {
        String details = "Stockpile - " + status.toString() + " " + Strings.nullToEmpty(additional);
        switch (status.getCode()) {
            case METRIC_NOT_FOUND:
                return SimulationStatus.NO_METRICS.withMessage(details);
            case INVALID_REQUEST:
                return SimulationStatus.INVALID_REQUEST.withMessage(details);
            default:
                return SimulationStatus.DATA_LOAD_ERROR.withMessage(details);
        }
    }

    public TSimulateEvaluationResponse toProto() {
        if (code != SimulationStatus.OK) {
            return TSimulateEvaluationResponse.newBuilder()
                .setRequestStatus(mapToRequestStatus())
                .setStatusMessage(message)
                .build();
        }

        if (resultingLines.isEmpty()) {
            return TSimulateEvaluationResponse.newBuilder()
                .setRequestStatus(ERequestStatusCode.INVALID_REQUEST)
                .setStatusMessage("No lines produced by request")
                .build();
        }

        var builder = TSimulateEvaluationResponse.newBuilder()
                .setRequestStatus(ERequestStatusCode.OK);

        var firstLine = resultingLines.get(0);
        var iterator = firstLine.getSource().iterator();
        RecyclableAggrPoint point = RecyclableAggrPoint.newInstance();
        while (iterator.next(point)) {
            builder.addTimeMillis(point.getTsMillis());
        }
        point.recycle();

        int pointsCount = builder.getTimeMillisCount();

        for (var line : resultingLines) {
            TAlertDataSeries.Builder alertDataSeries = TAlertDataSeries.newBuilder()
                    .setAlias(line.getAlias())
                    .addAllLabels(LabelConverter.labelsToProtoList(line.getLabels()));

            copy(line.getSource(), alertDataSeries);
            if (alertDataSeries.getValuesCount() != pointsCount) {
                return TSimulateEvaluationResponse.newBuilder()
                    .setRequestStatus(ERequestStatusCode.INTERNAL_ERROR)
                    .setStatusMessage("Different lines have different number of points")
                    .build();
            }

            builder.addAggregatedSeries(alertDataSeries);
        }

        for (var status : statuses) {
            builder.addEvaluatedStatuses(TEvaluationStatus.ECode.forNumber(status.getNumber()));
        }

        if (builder.getEvaluatedStatusesCount() != pointsCount) {
            return TSimulateEvaluationResponse.newBuilder()
                .setRequestStatus(ERequestStatusCode.INTERNAL_ERROR)
                .setStatusMessage("Number of points does not match the number of statuses")
                .build();
        }

        return builder.build();
    }

    private void copy(AggrGraphDataIterable source, TAlertDataSeries.Builder alertDataSeries) {
        RecyclableAggrPoint point = RecyclableAggrPoint.newInstance();
        var iterator = source.iterator();

        while (iterator.next(point)) {
            alertDataSeries.addValues(point.getValueDivided());
        }

        point.recycle();
    }

    private ERequestStatusCode mapToRequestStatus() {
        switch (code) {
            case OK:
                return ERequestStatusCode.OK;
            case NO_SUBALERTS:
            case NO_METRICS:
                return ERequestStatusCode.NOT_FOUND;
            case UNSUPPORTED:
            case TOO_MANY_METRICS:
            case INVALID_REQUEST:
                return ERequestStatusCode.INVALID_REQUEST;
            case GENERIC_ERROR:
            case DATA_LOAD_ERROR:
            case UNKNOWN:
            default:
                return ERequestStatusCode.INTERNAL_ERROR;
        }
    }

}
