package ru.yandex.solomon.alert.rule.expression;

import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.function.Function;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableSet;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.misc.thread.WhatThreadDoes;
import ru.yandex.monlib.metrics.labels.Label;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.alert.EvaluationStatus;
import ru.yandex.solomon.alert.domain.AlertKey;
import ru.yandex.solomon.alert.domain.SubAlert;
import ru.yandex.solomon.alert.domain.expression.ExpressionAlert;
import ru.yandex.solomon.alert.evaluation.AlertRuleExecutorType;
import ru.yandex.solomon.alert.rule.AbstractAlertRule;
import ru.yandex.solomon.alert.rule.AlertRuleDeadlines;
import ru.yandex.solomon.alert.rule.AlertTimeSeries;
import ru.yandex.solomon.alert.rule.ExplainResult;
import ru.yandex.solomon.alert.rule.MultipleTimeSeries;
import ru.yandex.solomon.alert.rule.PreloadedGraphDataLoader;
import ru.yandex.solomon.alert.rule.ResultOrProceed;
import ru.yandex.solomon.alert.rule.SimulationResult;
import ru.yandex.solomon.alert.rule.SimulationStatus;
import ru.yandex.solomon.alert.rule.usage.AlertRuleMetrics;
import ru.yandex.solomon.alert.statuses.AlertingStatusesSelector;
import ru.yandex.solomon.alert.util.Converters;
import ru.yandex.solomon.common.RequestProducer;
import ru.yandex.solomon.expression.NamedGraphData;
import ru.yandex.solomon.expression.analytics.GraphDataLoadRequest;
import ru.yandex.solomon.expression.analytics.PrepareContext;
import ru.yandex.solomon.expression.analytics.PreparedProgram;
import ru.yandex.solomon.expression.analytics.Program;
import ru.yandex.solomon.expression.expr.IntrinsicsExternalizer;
import ru.yandex.solomon.expression.type.SelType;
import ru.yandex.solomon.expression.value.SelValue;
import ru.yandex.solomon.flags.FeatureFlagsHolder;
import ru.yandex.solomon.labels.query.Selector;
import ru.yandex.solomon.labels.query.SelectorType;
import ru.yandex.solomon.labels.query.Selectors;
import ru.yandex.solomon.labels.query.SelectorsBuilder;
import ru.yandex.solomon.math.protobuf.Operation;
import ru.yandex.solomon.math.protobuf.OperationDownsampling;
import ru.yandex.solomon.metrics.client.MetricsClient;
import ru.yandex.solomon.metrics.client.combined.FindAndReadManyRequest;
import ru.yandex.solomon.util.time.Interval;

import static java.util.concurrent.CompletableFuture.completedFuture;
import static java.util.stream.Collectors.collectingAndThen;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;
import static java.util.stream.Collectors.toUnmodifiableMap;
import static ru.yandex.solomon.alert.rule.AlertRuleSelectors.enrichProjectSelector;

/**
 * <p>Trigger {@link ru.yandex.solomon.alert.EvaluationStatus.Code#ALARM}
 * as only expression start return {@code ru.yandex.solomon.expression.value.SelValueBoolean#TRUE}.
 *
 * <p>Trigger {@link ru.yandex.solomon.alert.EvaluationStatus.Code#NO_DATA}
 * only if not able resolve metric by specified label selector - metric not exits.
 *
 * <p>Trigger {@link ru.yandex.solomon.alert.EvaluationStatus.Code#ERROR}
 * if not able process one of the metric.
 *
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public class ExpressionAlertRule extends AbstractAlertRule {
    private static final Logger logger = LoggerFactory.getLogger(ExpressionAlertRule.class);

    private final Program program;
    private final Map<Selectors, Selectors> overrideSelectors;
    private final ExpressionAlertTemplateProcessor templateProcessor;

    @Nullable
    private final AlertingStatusesSelector alertingStatuses;

    public ExpressionAlertRule(
            ExpressionAlert alert,
            MetricsClient cachingMetricsClient,
            AlertRuleMetrics metrics,
            ExpressionAlertTemplateProcessor templateProcessor,
            Program program,
            @Nullable AlertingStatusesSelector alertingStatuses,
            FeatureFlagsHolder featureFlags)
    {
        super(alert, cachingMetricsClient, metrics, featureFlags);
        if (!alert.getGroupByLabels().isEmpty()) {
            throw new UnsupportedOperationException("Not able create rule by parent for SubAlert: " + alert);
        }
        this.templateProcessor = templateProcessor;
        this.alertingStatuses = alertingStatuses;

        this.program = program;
        this.overrideSelectors = this.program.getProgramSelectors()
            .stream()
            .distinct()
            .collect(toUnmodifiableMap(Function.identity(), selectors -> enrichProjectSelector(alert, selectors)));
    }

    public ExpressionAlertRule(
            SubAlert subAlert,
            MetricsClient cachingMetricsClient,
            AlertRuleMetrics metrics,
            ExpressionAlertTemplateProcessor templateProcessor,
            Program program,
            @Nullable AlertingStatusesSelector alertingStatuses,
            FeatureFlagsHolder featureFlags)
    {
        super(subAlert, cachingMetricsClient, metrics, featureFlags);
        ExpressionAlert parent = (ExpressionAlert) subAlert.getParent();

        this.alertingStatuses = alertingStatuses;
        this.templateProcessor = templateProcessor;
        this.program = program;
        Map<String, Selector> override = makeGroupSelectors(ImmutableSet.copyOf(parent.getGroupByLabels()), subAlert.getGroupKey());
        this.overrideSelectors = this.program.getProgramSelectors()
            .stream()
            .collect(toMap(
                Function.identity(),
                s -> enrichProjectSelector(alert, parseSelectorAndOverride(s, override)),
                (left, right) -> left));
    }

    public List<Selectors> getAllSelectors() {
        return this.program.getProgramSelectors().stream()
                .map(selectors -> overrideSelectors.getOrDefault(selectors, selectors))
                .collect(toList());
    }

    private Map<String, Selector> makeGroupSelectors(ImmutableSet<String> groupKeys, Labels groupKey) {
        Map<String, Selector> selectors = new HashMap<>(groupKeys.size());
        for (String name : groupKeys) {
            Label label = groupKey.findByKey(name);
            final Selector selector;
            if (label == null) {
                selector = Selector.absent(name);
            } else {
                selector = Selector.exact(name, label.getValue());
            }
            selectors.put(name, selector);
        }
        return selectors;
    }

    private Selectors parseSelectorAndOverride(Selectors selectors, Map<String, Selector> override) {
        SelectorsBuilder result = Selectors.builder(selectors.size());
        Set<String> disableOverride = new HashSet<>();
        for (Selector selector : selectors) {
            if (override.containsKey(selector.getKey())) {
                if (exactOrAbsent(selector)) {
                    disableOverride.add(selector.getKey());
                } else {
                    continue;
                }
            }

            result.add(selector);
        }

        for (Map.Entry<String, Selector> entry : override.entrySet()) {
            if (!disableOverride.contains(entry.getKey())) {
                result.add(entry.getValue());
            }
        }
        return result.build();
    }

    private boolean exactOrAbsent(Selector selector) {
        SelectorType type = selector.getType();
        return type == SelectorType.EXACT || type == SelectorType.ABSENT;
    }

    @Override
    public CompletableFuture<EvaluationStatus> eval(Instant now, AlertRuleDeadlines deadlines) {
        metrics.evaluations.inc();
        return explain(now, deadlines)
            .thenApply(ExplainResult::getStatus);
    }

    @Override
    public CompletableFuture<SimulationResult> simulate(Instant from, Instant to, Duration gridStep, AlertRuleDeadlines deadlines) {
        return completedFuture(SimulationStatus.UNSUPPORTED.withMessage("Simulation is not implemented for expression alerts"));
    }

    @Override
    public CompletableFuture<ExplainResult> explain(Instant nowNotDelayed, AlertRuleDeadlines deadlines) {
        Instant now = nowNotDelayed.minusSeconds(alert.getDelaySeconds());
        Instant from = now.minus(period);
        if (logger.isDebugEnabled()) {
            logger.debug("Execute alert {} on window [{}; {}]", alert, from, now);
        }

        PreparedProgram preparedProgram = preparedProgram(new Interval(from, now));
        Collection<GraphDataLoadRequest> loadRequests = preparedProgram.getLoadRequests();

        return CompletableFutures.safeCall(() -> loadMetrics(loadRequests, deadlines))
                .thenApply(metricsMaybeTooMany -> {
                    var metricsOrResult = metricsMaybeTooMany.compose(listManyTs -> {
                        int totalMetrics = listManyTs.stream().mapToInt(manyTs -> manyTs.getTimeSeriesList().size()).sum();
                        if (totalMetrics > alert.getMetricsLimit()) {
                            return ResultOrProceed.ready(tooManyMetrics());
                        } else {
                            return ResultOrProceed.proceed(listManyTs);
                        }
                    });
                    if (metricsOrResult.hasResult()) {
                        return templateProcessor.processTemplate(nowNotDelayed, new ExpressionCheckResult(metricsOrResult.getResult(), Map.of()));
                    }
                    return explainProgram(now, preparedProgram, metricsOrResult.getIncomplete());
                })
                .exceptionally(e -> {
                    EvaluationStatus status = classifyError(getId(), e);
                    return templateProcessor.processTemplate(nowNotDelayed, new ExpressionCheckResult(status, Map.of()));
                });
    }

    private CompletableFuture<ResultOrProceed<EvaluationStatus, List<MultipleTimeSeries>>> loadMetrics(Collection<GraphDataLoadRequest> loadRequests, AlertRuleDeadlines deadlines) {
        long startNanos = System.nanoTime();
        try {
            return loadRequests.stream()
                    .map(loadRequest -> {
                        var overriddenSelectors = overrideSelectors.get(loadRequest.getSelectors());
                        var overriddenLoadRequest = overriddenSelectors == null ? loadRequest : loadRequest.toBuilder()
                                .setSelectors(overriddenSelectors)
                                .build();
                        return loadSingleRequest(loadRequest, overriddenLoadRequest, deadlines);
                    })
                    .collect(collectingAndThen(toList(), CompletableFutures::allOf))
                    .thenApply(ResultOrProceed.allOf(AbstractAlertRule::mergeStatuses))
                    .thenApply(allResults -> allResults.compose(loadResults -> loadResults.stream()
                                    .map(this::convertToManyTimeSeries)
                                    .collect(collectingAndThen(toList(), ResultOrProceed.allOf(AbstractAlertRule::mergeStatuses)))));
        } finally {
            long spendNanos = System.nanoTime() - startNanos;
            metrics.cpuTimeNanos.add(spendNanos);
            cpuTime.mark(spendNanos);
        }
    }

    private CompletableFuture<ResultOrProceed<EvaluationStatus, LoadRequestAndResult>> loadSingleRequest(
            GraphDataLoadRequest originalLoadRequest,
            GraphDataLoadRequest overriddenLoadRequest,
            AlertRuleDeadlines deadlines)
    {
        Interval interval = overriddenLoadRequest.getInterval();
        var request = FindAndReadManyRequest.newBuilder()
                .setSelectors(overriddenLoadRequest.getSelectors())
                .setMetabaseLimit(alert.getMetricsLimit())
                .setFromMillis(interval.getBeginMillis())
                .setToMillis(interval.getEndMillis())
                .setProducer(RequestProducer.SYSTEM)
                .addOperation(Operation.newBuilder()
                        .setDownsampling(OperationDownsampling.newBuilder()
                                .setGridMillis(overriddenLoadRequest.getGridMillis())
                                .setAggregation(Converters.aggregateFunctionToProto(overriddenLoadRequest.getAggregateFunction()))
                                .setFillOption(OperationDownsampling.FillOption.NULL))
                        .build())
                .setSoftDeadline(deadlines.softResolveDeadline())
                .setSoftReadDeadline(deadlines.softReadDeadline())
                .setDeadline(deadlines.hardDeadline());

        return findAndReadMany(request)
                .handle((response, ex) -> checkReadManyResponse(overriddenLoadRequest, response, ex)
                        .map(metrics -> new LoadRequestAndResult(originalLoadRequest, metrics)));
    }

    private PreparedProgram preparedProgram(Interval interval) {
        long startNanos = System.nanoTime();
        try {
            AlertKey alertKey = alert.getKey();
            IntrinsicsExternalizer.Builder externalizer = IntrinsicsExternalizer.newBuilder(interval);

            if (alertingStatuses != null) {
                externalizer.setAlertKey(alertKey.getAlertId(), alertKey.getProjectId(), alertKey.getParentId());
                externalizer.setAlertingStatuses(alertingStatuses.getShard());
            }
            return program.prepare(PrepareContext.onInterval(interval)
                    .withExternalizer(externalizer.build())
                    .withSideEffectsProcessor(new EvaluationStatusSideEffectProcessor())
                    .build());
        } finally {
            long spendNanos = System.nanoTime() - startNanos;
            metrics.cpuTimeNanos.add(spendNanos);
            cpuTime.mark(spendNanos);
        }
    }

    private ExplainResult explainProgram(Instant now, PreparedProgram preparedProgram, List<MultipleTimeSeries> dataSet) {
        long startNanos = System.nanoTime();
        try {
            Map<String, SelValue> result = eval(preparedProgram, dataSet);
            EvaluationStatus status = makeEvaluationStatus(preparedProgram);
            List<AlertTimeSeries> series = makeTimeSeriesLines(result);
            return templateProcessor.processTemplate(now, new ExpressionCheckResult(status, result), series);
        } finally {
            long spendNanos = System.nanoTime() - startNanos;
            metrics.cpuTimeNanos.add(spendNanos);
            cpuTime.mark(spendNanos);
        }
    }

    private List<AlertTimeSeries> makeTimeSeriesLines(Map<String, SelValue> programResult) {
        List<AlertTimeSeries> result = new ArrayList<>(programResult.size());
        for (Map.Entry<String, SelValue> entry : programResult.entrySet()) {
            if (entry.getKey().indexOf('$') != -1) {
                continue;
            }
            SelValue value = entry.getValue();
            SelType type = value.type();
            if (type.isGraphData()) {
                result.add(makeTimeSeries(entry.getKey(), value));
            } else if (type.isVector() && type.vector().elementType.isGraphData()) {
                for (SelValue item : value.castToVector().valueArray()) {
                    result.add(makeTimeSeries(entry.getKey(), item));
                }
            }
        }

        return result;
    }

    private AlertTimeSeries makeTimeSeries(String alias, SelValue value) {
        NamedGraphData namedGraphData = value.castToGraphData().getNamedGraphData();
        Labels labels = namedGraphData.getLabels();
        var source = namedGraphData.getAggrGraphDataArrayList();
        return new AlertTimeSeries(alias, labels, namedGraphData.getDataType(), source);
    }

    private EvaluationStatus makeEvaluationStatus(PreparedProgram preparedProgram) {
        EvaluationStatusSideEffectProcessor processor = (EvaluationStatusSideEffectProcessor) preparedProgram.getSideEffectProcessor();

        return processor.getStatus();
    }

    private Map<String, SelValue> eval(PreparedProgram preparedProgram, List<MultipleTimeSeries> dataSet) {
        WhatThreadDoes.Handle h = WhatThreadDoes.push("Evaluate alert " + alert.getKey());
        try {
            PreloadedGraphDataLoader graphDataLoader = new PreloadedGraphDataLoader(dataSet);
            return preparedProgram.evaluate(graphDataLoader, Collections.emptyMap());
        } catch (RuntimeException e) {
            metrics.recordEvalFailure();
            throw e;
        } finally {
            h.popSafely();
        }
    }

    @Nullable
    @Override
    protected EvaluationStatus applyNoMetricsPolicy(Function<EvaluationStatus.Code, EvaluationStatus> statusMapper) {
        return switch (alert.getResolvedEmptyPolicy()) {
            case OK -> statusMapper.apply(EvaluationStatus.Code.OK);
            case WARN -> statusMapper.apply(EvaluationStatus.Code.WARN);
            case ALARM -> statusMapper.apply(EvaluationStatus.Code.ALARM);
            case DEFAULT, NO_DATA -> statusMapper.apply(EvaluationStatus.Code.NO_DATA);
            case MANUAL -> null;
        };
    }

    @Nullable
    @Override
    protected EvaluationStatus applyNoPointsPolicy(Function<EvaluationStatus.Code, EvaluationStatus> statusMapper) {
        return switch (alert.getNoPointsPolicy()) {
            case OK -> statusMapper.apply(EvaluationStatus.Code.OK);
            case WARN -> statusMapper.apply(EvaluationStatus.Code.WARN);
            case ALARM -> statusMapper.apply(EvaluationStatus.Code.ALARM);
            case NO_DATA -> statusMapper.apply(EvaluationStatus.Code.NO_DATA);
            case MANUAL, DEFAULT -> null;
        };
    }

    @Override
    public AlertRuleExecutorType getExecutorType() {
        if (getAllSelectors().stream().anyMatch(this::throughDataProxy)) {
            return AlertRuleExecutorType.DATAPROXY;
        }
        return AlertRuleExecutorType.DEFAULT;
    }
}
