package ru.yandex.solomon.alert.rule.expression;

import java.time.Instant;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Callable;
import java.util.function.IntFunction;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableMap;
import org.apache.commons.lang3.StringUtils;

import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.alert.EvaluationStatus;
import ru.yandex.solomon.alert.domain.expression.ExpressionAlert;
import ru.yandex.solomon.alert.evaluation.EvaluationStatusTemplateUtils;
import ru.yandex.solomon.alert.rule.AnnotationsTemplate;
import ru.yandex.solomon.alert.rule.TemplateProcessor;
import ru.yandex.solomon.alert.template.TemplateFactory;
import ru.yandex.solomon.expression.type.SelType;
import ru.yandex.solomon.expression.value.SelValue;
import ru.yandex.solomon.model.timeseries.GraphData;
import ru.yandex.solomon.util.collection.array.DoubleArrayView;
import ru.yandex.solomon.util.collection.array.LongArrayView;
import ru.yandex.solomon.util.time.InstantUtils;

/**
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public class ExpressionAlertTemplateProcessor implements TemplateProcessor<ExpressionCheckResult> {
    private final ExpressionAlert alert;
    private final Labels labels;
    private final AnnotationsTemplate annotationsTemplate;
    private final AnnotationsTemplate serviceProviderAnnotationsTemplate;

    public ExpressionAlertTemplateProcessor(ExpressionAlert alert, TemplateFactory templateFactory, Labels labels) {
        this.alert = alert;
        this.annotationsTemplate = new AnnotationsTemplate(alert.getAnnotations(), templateFactory);
        this.serviceProviderAnnotationsTemplate = new AnnotationsTemplate(alert.getServiceProviderAnnotations(), templateFactory);
        this.labels = labels;
    }

    @Override
    public EvaluationStatus processResult(Instant now, ExpressionCheckResult result) {
        EvaluationStatus status = result.status().withScalars(extractScalars(result.variables()));
        if (annotationsTemplate.isEmpty() && serviceProviderAnnotationsTemplate.isEmpty()) {
            return status;
        }

        Map<String, Object> params = ImmutableMap.<String, Object>builder()
                .put("alert", alert)
                .put("fromTime", now.minus(alert.getPeriod()))
                .put("toTime", now)
                .put("labels", labels.toMap())
                .put("labelsString", labels.toString())
                .put("status", ImmutableMap.of("code", status.getCode().name(), "details", status.getDescription()))
                .put(EvaluationStatusTemplateUtils.statusToTemplateKey(status.getCode()), true)
                .put("expression", expressionParamsForTemplate(result.variables()))
                .build();

        if (!annotationsTemplate.isEmpty()) {
            Map<String, String> annotations = annotationsTemplate.process(params);
            if (StringUtils.isNotEmpty(status.getDescription())) {
                annotations.putIfAbsent("causedBy", status.getDescription());
            }
            status = status.withAnnotations(annotations);
        }

        if (!serviceProviderAnnotationsTemplate.isEmpty()) {
            Map<String, String> annotations = serviceProviderAnnotationsTemplate.process(params);
            status = status.withServiceProviderAnnotations(annotations);
        }

        return status;
    }

    private Map<String, Object> expressionParamsForTemplate(Map<String, SelValue> variables) {
        Map<String, Object> result = new HashMap<>(variables.size());
        for (Map.Entry<String, SelValue> entry : variables.entrySet()) {
            Callable<Object> lazy = () -> unwrap(entry.getValue());
            result.put(entry.getKey(), lazy);
        }

        return result;
    }

    private static class LongListFormatter {
        final int length;
        final IntFunction<String> elemGetter;

        private String prefix = "[";
        private String suffix = "]";
        private int keepFront = 5;
        private int keepBack = 5;
        private String elemKind = "";

        private LongListFormatter(int length, IntFunction<String> elemGetter) {
            this.length = length;
            this.elemGetter = elemGetter;
        }

        public static LongListFormatter of(int length, IntFunction<String> elemGetter) {
            return new LongListFormatter(length, elemGetter);
        }

        public static LongListFormatter of(double[] array) {
            return new LongListFormatter(array.length, i -> String.valueOf(array[i]));
        }

        public LongListFormatter setPrefix(String prefix) {
            this.prefix = prefix;
            return this;
        }

        public LongListFormatter setSuffix(String suffix) {
            this.suffix = suffix;
            return this;
        }

        public LongListFormatter setKeepFront(int keepFront) {
            this.keepFront = keepFront;
            return this;
        }

        public LongListFormatter setKeepBack(int keepBack) {
            this.keepBack = keepBack;
            return this;
        }

        public LongListFormatter setElemKind(String elemKind) {
            this.elemKind = elemKind;
            return this;
        }

        private static void appendJoining(int fromInc, int toExcl, IntFunction<String> getter, String delim, StringBuilder sb) {
            if (fromInc >= toExcl) {
                return;
            }

            sb.append(getter.apply(fromInc));

            for (int i = fromInc + 1; i < toExcl; i++) {
                sb.append(delim);
                sb.append(getter.apply(i));
            }
        }

        public String format() {
            StringBuilder sb = new StringBuilder(prefix);
            if (keepFront + keepBack < length) {
                int skipCount = length - keepFront - keepBack;
                String skip = skipCount + (elemKind.isEmpty() ? " more" : " more " + elemKind + (skipCount > 1 ? "s" : ""));

                appendJoining(0, keepFront, elemGetter, ", ", sb);
                sb.append(", <").append(skip).append(">, ");
                appendJoining(length - keepBack, length, elemGetter, ", ", sb);
            } else {
                appendJoining(0, length, elemGetter, ", ", sb);
            }
            sb.append(suffix);
            return sb.toString();
        }
    }

    @Nullable
    private Object unwrap(SelValue variable) {
        SelType type = variable.type();
        if (type.isString()) {
            return variable.castToString().getValue();
        } else if (type.isDouble()) {
            return variable.castToScalar().getValue();
        } else if (type.isBoolean()) {
            return variable.castToBoolean().getValue();
        } else if (type.isGraphData()) {
            GraphData gd = variable.castToGraphData().getGraphData();
            LongArrayView ts = gd.getTimestamps();
            DoubleArrayView values = gd.getValues();
            return LongListFormatter.of(ts.length(), i -> InstantUtils.formatToMillis(ts.at(i)))
                    .setPrefix("Timeseries{timestamps=[")
                    .format() +
                   LongListFormatter.of(values.length(), i -> String.valueOf(values.at(i)))
                    .setPrefix(", values=[").setSuffix("]}")
                    .format();
        } else if (type.isDuration()) {
            return variable.castToDuration().getDuration();
        } else if (type.isInterval()) {
            return variable.castToInterval().getInterval();
        } else if (type.isDoubleVector()) {
            double[] values = variable.castToVector().doubleArray();
            return LongListFormatter.of(values)
                    .setElemKind("value")
                    .format();
        } else if (type.isVector()) {
            SelValue[] values = variable.castToVector().valueArray();
            return LongListFormatter.of(values.length, i -> String.valueOf(unwrap(values[i])))
                    .setKeepFront(3).setKeepBack(2)
                    .setElemKind("element")
                    .format();
        }

        return null;
    }

    private List<EvaluationStatus.ScalarValue> extractScalars(Map<String, SelValue> variables) {
        var list = new ArrayList<EvaluationStatus.ScalarValue>(variables.size());
        for (Map.Entry<String, SelValue> entry : variables.entrySet()) {
            SelType type = entry.getValue().type();
            if (type.isString()) {
                list.add(new EvaluationStatus.ScalarValue(entry.getKey(), unwrap(entry.getValue()), EvaluationStatus.ScalarType.STRING));
            } else if (type.isDouble()) {
                list.add(new EvaluationStatus.ScalarValue(entry.getKey(), unwrap(entry.getValue()), EvaluationStatus.ScalarType.DOUBLE));
            } else if (type.isBoolean()) {
                list.add(new EvaluationStatus.ScalarValue(entry.getKey(), unwrap(entry.getValue()), EvaluationStatus.ScalarType.BOOLEAN));
            }
        }
        return list;
    }
}
