package ru.yandex.solomon.alert.rule.threshold;

import java.util.Collection;
import java.util.EnumMap;
import java.util.Map;
import java.util.function.ToDoubleFunction;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.alert.domain.threshold.ThresholdType;
import ru.yandex.solomon.math.protobuf.Aggregation;
import ru.yandex.solomon.model.point.AggrPoint;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
class DownsamplingAndAggregation {
    private final Aggregation downsampling;
    private final ToDoubleFunction<Collection<? extends AggrPoint>> aggregationOverIntervals;

    private final static EnumMap<ThresholdType, ToDoubleFunction<Collection<? extends AggrPoint>>> WINDOW_AGGREGATIONS = new EnumMap<>(Map.of(
        ThresholdType.LAST_NON_NAN, DownsamplingAndAggregation::lastNonNan,
        ThresholdType.AVG, DownsamplingAndAggregation::avg,
        ThresholdType.MIN, DownsamplingAndAggregation::min,
        ThresholdType.MAX, DownsamplingAndAggregation::max,
        ThresholdType.SUM, DownsamplingAndAggregation::sum
        // COUNT is not used since counting is done by Stockpile downsampling
        // @see ru.yandex.solomon.alert.rule.threshold.ThresholdAlertSimulator.SUPPORTED_AGGREGATIONS
    ));

    // Below are non-shortcut versions of WindowCheckFunctions

    private static double lastNonNan(Collection<? extends AggrPoint> window) {
        for (var point : window) {
            double val = point.getValueDivided();
            if (Double.isNaN(val)) {
                continue;
            }
            return val;
        }

        return Double.NaN;
    }

    private static double min(Collection<? extends AggrPoint> window) {
        double result = Double.NaN;

        for (var point : window) {
            double val = point.getValueDivided();
            if (Double.isNaN(val)) {
                continue;
            }
            if (Double.isNaN(result) || result > val) {
                result = val;
            }
        }

        return result;
    }

    private static double max(Collection<? extends AggrPoint> window) {
        double result = Double.NaN;

        for (var point : window) {
            double val = point.getValueDivided();
            if (Double.isNaN(val)) {
                continue;
            }
            if (Double.isNaN(result) || result < val) {
                result = val;
            }
        }

        return result;
    }

    private static double sum(Collection<? extends AggrPoint> window) {
        double sum = 0;
        int count = 0;

        for (var point : window) {
            double val = point.getValueDivided();
            if (Double.isNaN(val)) {
                continue;
            }
            count++;
            sum += val;
        }

        return count == 0 ? Double.NaN : sum;
    }

    private static double avg(Collection<? extends AggrPoint> window) {
        double sum = 0;
        int count = 0;

        for (var point : window) {
            double val = point.getValueDivided();
            if (Double.isNaN(val)) {
                continue;
            }
            count++;
            sum += val;
        }

        return count == 0 ? Double.NaN : sum / count;
    }

    public DownsamplingAndAggregation(
            Aggregation downsampling,
            ThresholdType aggregationOverIntervals)
    {
        this.downsampling = downsampling;
        this.aggregationOverIntervals = WINDOW_AGGREGATIONS.get(aggregationOverIntervals);
    }

    public Aggregation getDownsampling() {
        return downsampling;
    }

    public ToDoubleFunction<Collection<? extends AggrPoint>> getWindowAggregation() {
        return aggregationOverIntervals;
    }
}
