package ru.yandex.solomon.alert.rule.threshold;

import java.util.Objects;
import java.util.Optional;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;
import javax.annotation.ParametersAreNullableByDefault;

import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.alert.EvaluationStatus;
import ru.yandex.solomon.alert.domain.threshold.PredicateRule;
import ru.yandex.solomon.alert.domain.threshold.PredicateStatusResult;
import ru.yandex.solomon.alert.rule.AlertTimeSeries;

/**
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public class MetricCheckResult implements Comparable<MetricCheckResult> {
    private final EvaluationStatus status;

    @Nullable
    private final AlertTimeSeries timeSeries;
    @Nullable
    private final PredicateRule rule;
    private final long timeMillis;
    private final double value;

    private MetricCheckResult(@Nonnull Builder builder) {
        this.status = Objects.requireNonNull(builder.status, "status");
        this.timeSeries = builder.timeSeries;
        this.rule = builder.rule;
        this.timeMillis = builder.timeMillis;
        this.value = builder.value;
    }

    static Builder newBuilder() {
        return new Builder();
    }

    public static MetricCheckResult of(@Nonnull EvaluationStatus status) {
        return newBuilder()
                .withEvaluationStatus(status)
                .build();
    }

    @ParametersAreNullableByDefault
    public static class Builder {
        private EvaluationStatus status;

        private PredicateRule rule = null;
        private long timeMillis = 0;
        private double value = Double.NaN;
        private AlertTimeSeries timeSeries = null;

        Builder withTimeSeries(@Nonnull AlertTimeSeries timeSeries) {
            this.timeSeries = timeSeries;
            return this;
        }

        public Builder withEvaluationStatus(@Nonnull EvaluationStatus status) {
            this.status = status;
            return this;
        }

        Builder withPredicateRule(@Nullable PredicateRule rule) {
            this.rule = rule;
            if (rule != null) {
                this.withEvaluationStatus(rule.getTargetStatus().toEvaluationStatus());
            }

            return this;
        }

        Builder withTimestamp(long timeMillis) {
            this.timeMillis = timeMillis;
            return this;
        }

        Builder withValue(double value) {
            this.value = value;
            return this;
        }

        Builder withResult(@Nonnull PredicateStatusResult result) {
            return this
                .withTimestamp(result.getTimestamp())
                .withValue(result.getValue());
        }

        MetricCheckResult build() {
            return new MetricCheckResult(this);
        }
    }

    public Optional<Labels> getLabels() {
        return getTimeSeries().map(AlertTimeSeries::getLabels);
    }

    public EvaluationStatus.Code getStatusCode() {
        return status.getCode();
    }

    public EvaluationStatus.ErrorCode getStatusErrorCode() {
        return status.getErrorCode();
    }

    public long getTimeMillis() {
        return timeMillis;
    }

    public double getValue() {
        return value;
    }

    public EvaluationStatus getEvaluationStatus() {
        return status;
    }

    public Optional<AlertTimeSeries> getTimeSeries() {
        return Optional.ofNullable(timeSeries);
    }

    public Optional<PredicateRule> getPredicateRule() {
        return Optional.ofNullable(rule);
    }

    @Override
    public int compareTo(MetricCheckResult o) {
        if (getStatusCode().ordinal() < o.getStatusCode().ordinal()) {
            return -1;
        }
        if (getStatusCode().ordinal() > o.getStatusCode().ordinal()) {
            return 1;
        }
        return getStatusErrorCode().ordinal() - o.getStatusErrorCode().ordinal();
    }
}
