package ru.yandex.solomon.alert.rule.threshold;

import java.util.List;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.alert.EvaluationStatus;
import ru.yandex.solomon.alert.domain.NoPointsPolicy;
import ru.yandex.solomon.alert.domain.threshold.PredicateRule;
import ru.yandex.solomon.alert.domain.threshold.PredicateStatusResult;
import ru.yandex.solomon.alert.rule.AlertTimeSeries;
import ru.yandex.solomon.model.point.AggrPoint;
import ru.yandex.solomon.model.protobuf.MetricType;
import ru.yandex.solomon.model.timeseries.AggrGraphDataListIterator;
import ru.yandex.solomon.model.timeseries.FilteringAggrGraphDataIterator;
import ru.yandex.solomon.model.timeseries.MetricTypeTransfers;

import static java.util.stream.Collectors.toList;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public class MultiplePredicateChecker {

    private static AggrGraphDataListIterator convertKindAndFilterOutNans(AlertTimeSeries timeSeries) {
        var sourceIterator = timeSeries.getSource().iterator();
        if (timeSeries.getSource().isEmpty()) {
            return sourceIterator;
        }
        var iterator = MetricTypeTransfers.of(timeSeries.getDataType(), MetricType.DGAUGE, sourceIterator);
        return FilteringAggrGraphDataIterator.of(iterator, (mask, point) -> !Double.isNaN(point.valueNum));
    }

    private static MetricCheckResult noPointsStatus(EvaluationStatus status, AlertTimeSeries timeSeries) {
        return MetricCheckResult.newBuilder()
                .withTimeSeries(timeSeries)
                .withEvaluationStatus(status.withDescription("No points in metric " + timeSeries.getLabels().toString()))
                .build();
    }

    public static MetricCheckResult checkMultipleFunctions(NoPointsPolicy noPointsPolicy, List<WindowCheckFunction> checkFunctions, AlertTimeSeries timeSeries) {
        AggrGraphDataListIterator iterator = convertKindAndFilterOutNans(timeSeries);

        List<WindowCheckInstance> checkFunctionInstances = checkFunctions.stream()
            .map(WindowCheckFunction::makeInstance)
            .collect(toList());

        AggrPoint point = new AggrPoint(iterator.columnSetMask());
        boolean empty = true;
        while (iterator.next(point)) {
            empty = false;
            long ts = point.getTsMillis();
            double value = point.getValueDivided();

            boolean allPrevAreNotTriggered = true;

            @Nullable PredicateStatusResult maybeResult = PredicateStatusResult.NOT_TRIGGERED;
            @Nullable PredicateRule lastPredicateRule = null;
            for (WindowCheckInstance instance : checkFunctionInstances) {
                maybeResult = instance.consume(ts, value);
                lastPredicateRule = instance.getPredicateRule();
                if (maybeResult == null) {
                    allPrevAreNotTriggered = false;
                } else {
                    if (allPrevAreNotTriggered && maybeResult.isTriggered()) {
                        return MetricCheckResult.newBuilder()
                            .withTimeSeries(timeSeries)
                            .withPredicateRule(lastPredicateRule)
                            .withResult(maybeResult)
                            .build();
                    }
                }
            }

            if (allPrevAreNotTriggered) {
                return MetricCheckResult.newBuilder()
                    .withTimeSeries(timeSeries)
                    .withPredicateRule(lastPredicateRule)
                    .withEvaluationStatus(EvaluationStatus.OK)
                    .withResult(maybeResult)
                    .build();
            }
        }

        if (empty) {
            switch (noPointsPolicy) {
                case OK:
                    return noPointsStatus(EvaluationStatus.OK, timeSeries);
                case WARN:
                    return noPointsStatus(EvaluationStatus.WARN, timeSeries);
                case ALARM:
                    return noPointsStatus(EvaluationStatus.ALARM, timeSeries);
                case NO_DATA:
                case DEFAULT:
                    return noPointsStatus(EvaluationStatus.NO_DATA, timeSeries);
                case MANUAL:
                    // pass empty series to predicates, e.g. COUNT
                    break;
                default:
                    throw new IllegalArgumentException("Unsupported no points policy: " + noPointsPolicy);
            }
        }

        PredicateStatusResult result = PredicateStatusResult.NOT_TRIGGERED;
        @Nullable PredicateRule lastPredicateRule = null;
        for (WindowCheckInstance instance : checkFunctionInstances) {
            result = instance.getResult();
            lastPredicateRule = instance.getPredicateRule();
            if (result.isTriggered()) {
                return MetricCheckResult.newBuilder()
                    .withTimeSeries(timeSeries)
                    .withPredicateRule(lastPredicateRule)
                    .withResult(result)
                    .build();
            }
        }
        return MetricCheckResult.newBuilder()
            .withTimeSeries(timeSeries)
            .withPredicateRule(lastPredicateRule)
            .withEvaluationStatus(EvaluationStatus.OK)
            .withResult(result)
            .build();

    }

}
