package ru.yandex.solomon.alert.rule.threshold;

import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableSet;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.solomon.alert.EvaluationStatus;
import ru.yandex.solomon.alert.domain.SubAlert;
import ru.yandex.solomon.alert.domain.threshold.ThresholdAlert;
import ru.yandex.solomon.alert.evaluation.AlertRuleExecutorType;
import ru.yandex.solomon.alert.rule.AbstractAlertRule;
import ru.yandex.solomon.alert.rule.AlertRuleDeadlines;
import ru.yandex.solomon.alert.rule.AlertTimeSeries;
import ru.yandex.solomon.alert.rule.ExplainResult;
import ru.yandex.solomon.alert.rule.MultipleTimeSeries;
import ru.yandex.solomon.alert.rule.ResultOrProceed;
import ru.yandex.solomon.alert.rule.SimulationResult;
import ru.yandex.solomon.alert.rule.usage.AlertRuleMetrics;
import ru.yandex.solomon.alert.util.Converters;
import ru.yandex.solomon.common.RequestProducer;
import ru.yandex.solomon.expression.analytics.GraphDataLoadRequest;
import ru.yandex.solomon.expression.analytics.Program;
import ru.yandex.solomon.flags.FeatureFlagsHolder;
import ru.yandex.solomon.labels.query.Selectors;
import ru.yandex.solomon.math.protobuf.Operation;
import ru.yandex.solomon.math.protobuf.OperationDownsampling;
import ru.yandex.solomon.metrics.client.MetricsClient;
import ru.yandex.solomon.metrics.client.combined.FindAndReadManyRequest;
import ru.yandex.solomon.util.time.Interval;

import static ru.yandex.solomon.alert.rule.AlertRuleSelectors.enrichProjectSelector;
import static ru.yandex.solomon.alert.rule.AlertRuleSelectors.overrideMultiAlertSelector;
import static ru.yandex.solomon.alert.rule.AlertRuleSelectors.toOldFormat;

/**
 * Trigger alarm as only one of the timeseries resolved by selector violate threshold.
 *
 * <p>Trigger {@link EvaluationStatus.Code#ALARM} as only one of the metric violate threshold.
 * <p>Trigger {@link EvaluationStatus.Code#NO_DATA} if by specified selector not exists metrics,
 * or if all metrics resolved by selector doesn't have points on checked period.
 * <p>Trigger {@link EvaluationStatus.Code#ERROR} if not able process one of the metric,
 * but {@link EvaluationStatus.Code#ALARM} has a higher priority.
 *
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public class ThresholdAlertRule extends AbstractAlertRule {
    private static final Logger logger = LoggerFactory.getLogger(ThresholdAlertRule.class);

    private final List<WindowCheckFunction> timeSeriesCheckFunctions;
    private final Selectors selector;
    private final Transformer transformer;
    private final ThresholdAlertTemplateProcessor templateProcessor;
    private final ThresholdAlertSimulator simulator;

    public ThresholdAlertRule(
            ThresholdAlert alert,
            MetricsClient cachingMetricsClient,
            AlertRuleMetrics metrics,
            ThresholdAlertTemplateProcessor templateProcessor,
            Program program,
            FeatureFlagsHolder featureFlags)
    {
        super(alert, cachingMetricsClient, metrics, featureFlags);
        if (!alert.getGroupByLabels().isEmpty()) {
            throw new UnsupportedOperationException("Not able create rule by parent for SubAlert: " + alert);
        }
        this.timeSeriesCheckFunctions = alert.getPredicateRules().stream()
                .map(WindowCheckFunctionFactory::prepare)
                .collect(Collectors.toList());
        this.selector = toOldFormat(enrichProjectSelector(alert, alert.getSelectors()));
        this.templateProcessor = templateProcessor;
        this.transformer = new Transformer(alert.getTransformations(), program);
        this.simulator = new ThresholdAlertSimulator(alert, selector, transformer, cachingMetricsClient);
    }

    public ThresholdAlertRule(
            SubAlert subAlert,
            MetricsClient cachingMetricsClient,
            AlertRuleMetrics metrics,
            ThresholdAlertTemplateProcessor templateProcessor,
            Program program,
            FeatureFlagsHolder featureFlags)
    {
        super(subAlert, cachingMetricsClient, metrics, featureFlags);
        ThresholdAlert parent = (ThresholdAlert) subAlert.getParent();
        this.timeSeriesCheckFunctions = parent.getPredicateRules().stream()
                .map(WindowCheckFunctionFactory::prepare)
                .collect(Collectors.toList());
        this.selector = toOldFormat(enrichProjectSelector(subAlert, makeGroupSelector(subAlert, parent)));
        this.templateProcessor = templateProcessor;
        this.transformer = new Transformer(parent.getTransformations(), program);
        this.simulator = new ThresholdAlertSimulator(alert, selector, transformer, cachingMetricsClient);
    }

    private Selectors makeGroupSelector(SubAlert subAlert, ThresholdAlert parent) {
        return overrideMultiAlertSelector(
                ImmutableSet.copyOf(parent.getGroupByLabels()),
                subAlert.getGroupKey(),
                parent.getSelectors());
    }

    public Selectors getSelectors() {
        return selector;
    }

    @Override
    public CompletableFuture<EvaluationStatus> eval(Instant now, AlertRuleDeadlines deadlines) {
        this.metrics.evaluations.inc();
        return explainImpl(now, deadlines, false)
            .thenApply(ExplainResult::getStatus);
    }

    @Override
    public CompletableFuture<ExplainResult> explain(Instant now, AlertRuleDeadlines deadlines) {
        return explainImpl(now, deadlines, true);
    }

    @Override
    public CompletableFuture<SimulationResult> simulate(Instant from, Instant to, Duration gridStep, AlertRuleDeadlines deadlines) {
        return simulator.simulate(from, to, gridStep, deadlines);
    }

    private EvaluationStatus emptyTransformedResult(EvaluationStatus.Code statusCode) {
        return statusCode.toStatus("Empty set of lines after transformation");
    }

    @Nullable
    @Override
    protected EvaluationStatus applyNoMetricsPolicy(Function<EvaluationStatus.Code, EvaluationStatus> statusMapper) {
        return switch (alert.getResolvedEmptyPolicy()) {
            case OK -> statusMapper.apply(EvaluationStatus.Code.OK);
            case WARN -> statusMapper.apply(EvaluationStatus.Code.WARN);
            case ALARM -> statusMapper.apply(EvaluationStatus.Code.ALARM);
            case DEFAULT, NO_DATA -> statusMapper.apply(EvaluationStatus.Code.NO_DATA);
            case MANUAL -> throw new IllegalArgumentException("Unsupported resolved empty policy: " + alert.getResolvedEmptyPolicy());
        };
    }

    @Nullable
    @Override
    protected EvaluationStatus applyNoPointsPolicy(Function<EvaluationStatus.Code, EvaluationStatus> statusMapper) {
        // Threshold alert does emptiness checking in MultiplePredicateChecker, NOP here
        return null;
    }

    private CompletableFuture<ExplainResult> explainImpl(Instant nowNotDelayed, AlertRuleDeadlines deadlines, boolean withSources) {
        final Instant now = nowNotDelayed.minusSeconds(alert.getDelaySeconds());
        if (logger.isDebugEnabled()) {
            logger.debug("Execute alert {} on window [{}; {}]", alert, now.minus(period), now);
        }

        PreparedTransformer prepared = transformer.prepare(new Interval(now.minus(period), now));
        GraphDataLoadRequest loadRequest = prepared.getLoadRequest().toBuilder()
                .setSelectors(selector)
                .build();

        return CompletableFutures.safeCall(() -> loadMetrics(loadRequest, deadlines))
                .thenApply(metricsOrResult -> {
                    if (metricsOrResult.hasResult()) {
                        return templateProcessor.processTemplate(now, MetricCheckResult.of(metricsOrResult.getResult()));
                    }
                    return explainMetrics(prepared, now, metricsOrResult.getIncomplete(), withSources);
                })
                .exceptionally(e -> {
                    EvaluationStatus status = classifyError(getId(), e);
                    return templateProcessor.processTemplate(now, MetricCheckResult.of(status));
                });
    }

    private CompletableFuture<ResultOrProceed<EvaluationStatus, MultipleTimeSeries>> loadMetrics(
            GraphDataLoadRequest loadRequest,
            AlertRuleDeadlines deadlines)
    {
        long startNanos = System.nanoTime();
        try {
            Interval interval = loadRequest.getInterval();
            var request = FindAndReadManyRequest.newBuilder()
                    .setSelectors(selector)
                    .setMetabaseLimit(alert.getMetricsLimit())
                    .setFromMillis(interval.getBeginMillis())
                    .setToMillis(interval.getEndMillis())
                    .setProducer(RequestProducer.SYSTEM)
                    .addOperation(Operation.newBuilder()
                            .setDownsampling(OperationDownsampling.newBuilder()
                                    .setGridMillis(loadRequest.getGridMillis())
                                    .setAggregation(Converters.aggregateFunctionToProto(loadRequest.getAggregateFunction()))
                                    .setFillOption(OperationDownsampling.FillOption.NULL))
                            .build())
                    .setSoftDeadline(deadlines.softResolveDeadline())
                    .setSoftReadDeadline(deadlines.softReadDeadline())
                    .setDeadline(deadlines.hardDeadline());

            return findAndReadMany(request)
                    .handle((response, ex) -> checkReadManyResponse(loadRequest, response, ex)
                            .compose(metrics -> convertToManyTimeSeries(new LoadRequestAndResult(loadRequest, metrics))));
        } finally {
            long spendNanos = System.nanoTime() - startNanos;
            metrics.cpuTimeNanos.add(spendNanos);
            cpuTime.mark(spendNanos);
        }
    }

    private ExplainResult explainMetrics(
            PreparedTransformer prepared,
            Instant now,
            MultipleTimeSeries metrics,
            boolean withSources)
    {
        long startNanos = System.nanoTime();
        try {
            List<MetricCheckResult> results = prepared.transform(metrics.getTimeSeriesList()).stream()
                    .map(this::checkTimeSeries)
                    .collect(Collectors.toList());

            if (results.size() == 0) {
                EvaluationStatus status = applyNoMetricsPolicy(this::emptyTransformedResult);
                if (status != null) {
                    return templateProcessor.processTemplate(now, MetricCheckResult.of(status));
                }
            }

            MetricCheckResult merged = results.get(0);
            List<AlertTimeSeries> sources = withSources ? new ArrayList<>(results.size()) : List.of();
            for (MetricCheckResult result : results) {
                merged = mergeCheckResults(merged, result);
                if (withSources) {
                    result.getTimeSeries().ifPresent(sources::add);
                }
            }

            return templateProcessor.processTemplate(now, merged, sources);
        } finally {
            long spendNanos = System.nanoTime() - startNanos;
            super.metrics.cpuTimeNanos.add(spendNanos);
            cpuTime.mark(spendNanos);
        }
    }

    private MetricCheckResult mergeCheckResults(MetricCheckResult left, MetricCheckResult right) {
        if (left.compareTo(right) >= 0) {
            return left;
        } else {
            return right;
        }
    }

    private MetricCheckResult checkTimeSeries(AlertTimeSeries timeSeries) {
        return MultiplePredicateChecker.checkMultipleFunctions(alert.getNoPointsPolicy(), timeSeriesCheckFunctions, timeSeries);
    }

    @Override
    public AlertRuleExecutorType getExecutorType() {
        if (throughDataProxy(getSelectors())) {
            return AlertRuleExecutorType.DATAPROXY;
        }
        return AlertRuleExecutorType.DEFAULT;
    }
}
