package ru.yandex.solomon.alert.rule.threshold;

import java.time.Instant;
import java.util.Map;
import java.util.Optional;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableMap;
import org.apache.commons.lang3.StringUtils;

import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.alert.EvaluationStatus;
import ru.yandex.solomon.alert.domain.threshold.PredicateRule;
import ru.yandex.solomon.alert.domain.threshold.ThresholdAlert;
import ru.yandex.solomon.alert.evaluation.EvaluationStatusTemplateUtils;
import ru.yandex.solomon.alert.rule.AnnotationsTemplate;
import ru.yandex.solomon.alert.rule.TemplateProcessor;
import ru.yandex.solomon.alert.template.TemplateFactory;

/**
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public class ThresholdAlertTemplateProcessor implements TemplateProcessor<MetricCheckResult> {
    private final ThresholdAlert alert;
    private final AnnotationsTemplate annotationsTemplate;
    private final AnnotationsTemplate serviceProviderAnnotationsTemplate;

    public ThresholdAlertTemplateProcessor(ThresholdAlert alert, TemplateFactory templateFactory) {
        this.alert = alert;
        this.annotationsTemplate = new AnnotationsTemplate(alert.getAnnotations(), templateFactory);
        this.serviceProviderAnnotationsTemplate = new AnnotationsTemplate(alert.getServiceProviderAnnotations(), templateFactory);
    }

    @Override
    public EvaluationStatus processResult(Instant now, MetricCheckResult result) {
        var status = result.getEvaluationStatus();
        if (annotationsTemplate.isEmpty() && serviceProviderAnnotationsTemplate.isEmpty()) {
            return status;
        }

        Optional<PredicateRule> rule = result.getPredicateRule();

        var params = ImmutableMap.<String, Object>builder()
                .put("alert", alert)
                .put("fromTime", now.minus(alert.getPeriod()))
                .put("toTime", now)
                .put("status", ImmutableMap.of("code", status.getCode().name(), "details", status.getDescription()))
                .put("pointValue", result.getValue())
                .put("pointTime", Instant.ofEpochMilli(result.getTimeMillis()))
                .put(EvaluationStatusTemplateUtils.statusToTemplateKey(result.getStatusCode()), true);

        rule.map(PredicateRule::getThresholdType).ifPresent(value -> params.put("alert.aggregation", value));
        rule.map(PredicateRule::getComparison).ifPresent(value -> params.put("alert.comparison", value));
        rule.map(PredicateRule::getThreshold).ifPresent(value -> params.put("alert.threshold", value));
        result.getLabels().map(Labels::toMap).ifPresent(value -> params.put("labels", value));
        result.getLabels().map(Labels::toString).ifPresent(value -> params.put("labelsString", value));

        if (!annotationsTemplate.isEmpty()) {
            Map<String, String> annotations = annotationsTemplate.process(params.build());
            if (StringUtils.isNotEmpty(status.getDescription())) {
                annotations.putIfAbsent("description", status.getDescription());
            }
            status = status.withAnnotations(annotations);
        }

        if (!serviceProviderAnnotationsTemplate.isEmpty()) {
            Map<String, String> annotations = serviceProviderAnnotationsTemplate.process(params.build());
            status = status.withServiceProviderAnnotations(annotations);
        }

        return status;
    }
}
