package ru.yandex.solomon.alert.rule.threshold;

import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.alert.rule.AlertTimeSeries;
import ru.yandex.solomon.alert.rule.MultipleTimeSeries;
import ru.yandex.solomon.alert.rule.PreloadedGraphDataLoader;
import ru.yandex.solomon.alert.rule.SingleTimeSeries;
import ru.yandex.solomon.expression.NamedGraphData;
import ru.yandex.solomon.expression.analytics.GraphDataLoadRequest;
import ru.yandex.solomon.expression.analytics.GraphDataLoader;
import ru.yandex.solomon.expression.analytics.PreparedProgram;
import ru.yandex.solomon.expression.analytics.Program;
import ru.yandex.solomon.expression.type.SelTypes;
import ru.yandex.solomon.expression.value.SelValue;
import ru.yandex.solomon.expression.value.SelValueGraphData;
import ru.yandex.solomon.labels.query.Selectors;
import ru.yandex.solomon.util.time.Interval;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public class Transformer {
    private final String transformations;
    private final Program program;
    private final boolean grouping;

    public Transformer(String transformations, Program program) {
        this.transformations = transformations;
        this.program = program;

        this.grouping = computeIsGrouping(program);
    }

    class PreparedTransformerImpl implements PreparedTransformer {
        private final PreparedProgram preparedProgram;

        private PreparedTransformerImpl(PreparedProgram preparedProgram) {
            this.preparedProgram = preparedProgram;
        }

        @Override
        public boolean isGrouping() {
            return grouping;
        }

        @Override
        public GraphDataLoadRequest getLoadRequest() {
            var loadRequests = preparedProgram.getLoadRequests();
            if (loadRequests.size() != 1) {
                throw new IllegalArgumentException("Transformation should have single input time series");
            }
            return loadRequests.iterator().next();
        }

        @Override
        public List<AlertTimeSeries> transform(List<SingleTimeSeries> input) {
            GraphDataLoader graphDataLoader = new PreloadedGraphDataLoader(List.of(new MultipleTimeSeries(getLoadRequest(), input)));
            var vars = preparedProgram.evaluate(graphDataLoader, Map.of());
            String resultVar = preparedProgram.expressionToVarName(transformations);
            SelValue result = vars.get(resultVar);
            if (result.type().isGraphData()) {
                NamedGraphData ngd = result.castToGraphData().getNamedGraphData();
                return List.of(new AlertTimeSeries(ngd));
            } else if (result.type().isGraphDataVector()) {
                return Arrays.stream(result.castToVector().valueArray())
                    .map(SelValue::castToGraphData)
                    .map(SelValueGraphData::getNamedGraphData)
                    .map(AlertTimeSeries::new)
                    .collect(Collectors.toList());
            } else {
                throw new IllegalArgumentException("The result of transformation was neither GraphData nor Vector<GraphData> but " + result.type());
            }
        }
    }

    static class PreparedTransformerIdentity implements PreparedTransformer {
        final Interval interval;

        PreparedTransformerIdentity(Interval interval) {
            this.interval = interval;
        }

        @Override
        public boolean isGrouping() {
            return false;
        }

        @Override
        public GraphDataLoadRequest getLoadRequest() {
            return GraphDataLoadRequest.newBuilder(Selectors.of())
                    .setInterval(interval)
                    .setType(SelTypes.GRAPH_DATA_VECTOR)
                    .build();
        }

        @Override
        public List<AlertTimeSeries> transform(List<SingleTimeSeries> input) {
            return input.stream()
                    .map(SingleTimeSeries::castToAlertTimeSeries)
                    .collect(Collectors.toList());
        }
    }


    public PreparedTransformer prepare(Interval interval) {
        if (transformations.isEmpty()) {
            return new PreparedTransformerIdentity(interval);
        } else {
            return new PreparedTransformerImpl(program.prepare(interval));
        }
    }

    private static boolean computeIsGrouping(Program program) {
        // TODO(uranix): smart check, e.g. derivative -> non grouping, group_lines -> grouping, percentile_group_lines -> grouping, etc
        return !program.isEmpty();
    }
}
