package ru.yandex.solomon.alert.rule.threshold;

import java.util.DoubleSummaryStatistics;
import java.util.function.DoublePredicate;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.alert.domain.threshold.PredicateRule;

/**
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public final class WindowCheckFunctionFactory {
    private WindowCheckFunctionFactory() {
    }

    public static WindowCheckFunction prepare(PredicateRule predicateRule) {
        DoublePredicate predicate = makeDoublePredicate(predicateRule);
        switch (predicateRule.getThresholdType()) {
            case AT_LEAST_ONE:
                return new AtLeastOneWindowCheckFunction(predicate, predicateRule);
            case AT_ALL_TIMES:
                return new AtAllTimeWindowCheckFunction(predicate, predicateRule);
            case LAST_NON_NAN:
                return new LastNoNanWindowFunction(predicate, predicateRule);
            case AVG:
                return new AggregationWindowCheckFunction(predicate, DoubleSummaryStatistics::getAverage, predicateRule);
            case MAX:
                return new AggregationWindowCheckFunction(predicate, DoubleSummaryStatistics::getMax, predicateRule);
            case MIN:
                return new AggregationWindowCheckFunction(predicate, DoubleSummaryStatistics::getMin, predicateRule);
            case SUM:
                return new AggregationWindowCheckFunction(predicate, DoubleSummaryStatistics::getSum, predicateRule);
            case COUNT:
                return new AggregationWindowCheckFunction(predicate, DoubleSummaryStatistics::getCount, predicateRule);
            default:
                throw new UnsupportedOperationException("Unsupported window match for predicate rule: " + predicateRule);
        }
    }

    static DoublePredicate makeDoublePredicate(PredicateRule predicateRule) {
        final double threshold = predicateRule.getThreshold();
        switch (predicateRule.getComparison()) {
            case EQ:
                return value -> Double.compare(value, threshold) == 0;
            case NE:
                return value -> Double.compare(value, threshold) != 0;
            case LT:
                return value -> Double.compare(value, threshold) < 0;
            case LTE:
                return value -> Double.compare(value, threshold) <= 0;
            case GT:
                return value -> Double.compare(value, threshold) > 0;
            case GTE:
                return value -> Double.compare(value, threshold) >= 0;
            default:
                throw new UnsupportedOperationException("Predicate rule with unsupported compare type: " + predicateRule);
        }
    }
}
