package ru.yandex.solomon.alert.stats;

import java.time.Clock;
import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.util.concurrent.MoreExecutors;
import org.apache.commons.lang3.tuple.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.impl.ytree.object.annotation.YTreeObject;
import ru.yandex.inside.yt.kosher.impl.ytree.object.serializers.YTreeObjectSerializerFactory;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.alert.api.AlertingException;
import ru.yandex.solomon.alert.client.NotificationApi;
import ru.yandex.solomon.alert.cluster.broker.notification.ChannelValidationService;
import ru.yandex.solomon.alert.dao.ProjectHolderImpl;
import ru.yandex.solomon.alert.dao.ProjectsHolder;
import ru.yandex.solomon.alert.domain.AlertSeverity;
import ru.yandex.solomon.alert.protobuf.ERequestStatusCode;
import ru.yandex.solomon.alert.protobuf.TListNotificationsRequest;
import ru.yandex.solomon.alert.protobuf.notification.TNotification;
import ru.yandex.solomon.config.TimeUnitConverter;
import ru.yandex.solomon.config.protobuf.alert.ResourceMonitoringStatsConfig;
import ru.yandex.solomon.locks.DistributedLock;
import ru.yandex.solomon.locks.LockSubscriber;
import ru.yandex.solomon.locks.UnlockReason;
import ru.yandex.solomon.secrets.SecretProvider;
import ru.yandex.solomon.selfmon.counters.AsyncMetrics;
import ru.yandex.solomon.util.actors.AsyncActorBody;
import ru.yandex.solomon.util.actors.AsyncActorRunner;
import ru.yandex.solomon.util.actors.PingActorRunner;
import ru.yandex.solomon.util.time.DurationUtils;
import ru.yandex.yt.ytclient.proxy.TableWriter;
import ru.yandex.yt.ytclient.proxy.YtClient;
import ru.yandex.yt.ytclient.proxy.request.WriteTable;
import ru.yandex.yt.ytclient.rpc.RpcCredentials;

import static java.util.concurrent.CompletableFuture.completedFuture;

/**
 * @author Alexey Trushkin
 */
@ParametersAreNonnullByDefault
public class ChannelsMonitoringStatsExporter implements AutoCloseable {
    private static final Logger logger = LoggerFactory.getLogger(ChannelsMonitoringStatsExporter.class);
    private static final Duration deadline = Duration.ofSeconds(30);
    private static final int LIST_PAGE_SIZE = 3;

    private final DistributedLock lock;
    private final Clock clock;
    private final long initialDelayMillis;
    private final long minReindexPeriod;
    private final ProjectsHolder projectsHolder;
    private final NotificationApi notificationApi;
    private final ChannelValidationService channelValidationService;
    private final YtClient ytClient;
    private final ResourceMonitoringStatsConfig config;
    private volatile long latestReindexTs;
    private volatile boolean closed;
    private final ScheduledExecutorService timer;
    private final PingActorRunner actor;
    private final AsyncMetrics metrics;

    public ChannelsMonitoringStatsExporter(
            ProjectsHolder projectsHolder,
            DistributedLock lock,
            ExecutorService executor,
            ScheduledExecutorService timer,
            Clock clock,
            MetricRegistry registry,
            ResourceMonitoringStatsConfig config,
            SecretProvider secretProvider,
            NotificationApi notificationApi,
            ChannelValidationService channelValidationService)
    {
        this.projectsHolder = projectsHolder;
        this.notificationApi = notificationApi;
        this.channelValidationService = channelValidationService;
        var intervalMillis = TimeUnitConverter.millis(config.getResourceStatusRefreshInterval());
        this.lock = lock;
        this.clock = clock;
        this.initialDelayMillis = Duration.ofMinutes(1).toMillis();
        minReindexPeriod = intervalMillis / 2;
        this.timer = timer;
        this.metrics = new AsyncMetrics(registry, "resources.monitoring_stats.exporter", Labels.of("type", "channels"));
        this.ytClient = YtClient.builder()
                .setCluster(config.getYtCluster())
                .setRpcCredentials(new RpcCredentials("robot-solomon", secretProvider.getSecret(config.getOAuthToken()).orElseThrow()))
                .build();
        this.config = config;
        this.actor = PingActorRunner.newBuilder()
                .executor(executor)
                .timer(timer)
                .operation("Export monitoring project channels stats")
                .pingInterval(Duration.ofMillis(intervalMillis))
                .backoffDelay(Duration.ofMinutes(1))
                .onPing(this::act)
                .build();
        acquireLock();
    }

    public CompletableFuture<Void> exportStats() {
        var iterator = projectsHolder.getProjects().stream()
                .filter(project -> !config.getSkipYasm() || !project.startsWith("yasm_"))
                .iterator();
        var actorBody = new NotificationFetcher(iterator);
        var runner = new AsyncActorRunner(actorBody, MoreExecutors.directExecutor(), 200);
        return runner.start()
                .thenCompose(unused -> prepareResults(actorBody.results))
                .thenCompose(tableRows -> new YtTableWriter(tableRows).write());
    }

    private CompletableFuture<List<TableRow>> prepareResults(ConcurrentMap<String, List<TNotification>> notifications) {
        CompletableFuture<Void> done = CompletableFuture.completedFuture(null);
        List<TableRow> rows = new ArrayList<>(notifications.size());
        for (var id : notifications.keySet()) {
            done = done.thenCompose(unused -> channelValidationService.validateSeverities(notifications.get(id))
                    .thenAccept(result -> rows.add(createTableRow(id, result))));
        }
        return done
                .thenApply(unused -> rows);
    }

    private TableRow createTableRow(String id, ChannelValidationService.Result result) {
        TableRow r = new TableRow();
        Map<String, String> severity = new HashMap<>();
        Map<String, List<ChannelStatus>> channels = new HashMap<>();
        severity.put(AlertSeverity.DISASTER.name(), "false");
        severity.put(AlertSeverity.CRITICAL.name(), "false");
        severity.put(AlertSeverity.INFO.name(), "false");
        for (var entry : result.channels().entrySet()) {
            List<ChannelStatus> statuses = new ArrayList<>();
            for (ChannelValidationService.ResultRow resultRow : entry.getValue()) {
                if (resultRow.severity().equals(AlertSeverity.DISASTER) && resultRow.result()) {
                    severity.put(AlertSeverity.DISASTER.name(), "true");
                } else if (resultRow.severity().equals(AlertSeverity.CRITICAL) && resultRow.result()) {
                    severity.put(AlertSeverity.CRITICAL.name(), "true");
                } else if (resultRow.severity().equals(AlertSeverity.INFO) && resultRow.result()) {
                    severity.put(AlertSeverity.INFO.name(), "true");
                }
                statuses.add(new ChannelStatus("channel", resultRow.severity().name(), resultRow.error(), resultRow.result()));
            }
            channels.put(entry.getKey(), statuses);
        }
        r.abc_slug = projectsHolder.getProjectView(id).map(ProjectHolderImpl.ProjectView::abc).orElse("");
        r.project_id = id;
        r.severity_status= severity;
        r.channel_status = channels;
        return r;
    }

    public CompletableFuture<Void> act(int attempt) {
        if (closed) {
            return completedFuture(null);
        }

        if (!lock.isLockedByMe()) {
            return completedFuture(null);
        }
        long reindexTs = clock.millis();
        if (minReindexPeriod > reindexTs - latestReindexTs) {
            return completedFuture(null);
        }

        var future = exportStats()
                .thenAccept(ignore -> latestReindexTs = reindexTs);
        metrics.forFuture(future);
        return future;
    }

    public void schedule() {
        long delay = DurationUtils.randomize(initialDelayMillis);
        timer.schedule(actor::forcePing, delay, TimeUnit.MILLISECONDS);
    }

    private void acquireLock() {
        if (closed) {
            return;
        }

        lock.acquireLock(new LockSubscriber() {
            @Override
            public boolean isCanceled() {
                return closed;
            }

            @Override
            public void onLock(long seqNo) {
                logger.info("Acquire ChannelsMonitoringStatsExporterMaster lock, seqNo {}", seqNo);
                schedule();
            }

            @Override
            public void onUnlock(UnlockReason reason) {
                logger.info("Loose ChannelsMonitoringStatsExporterMaster lock by reason: {}", reason);
                acquireLock();
            }
        }, 5, TimeUnit.MINUTES);
    }

    @Override
    public void close() {
        closed = true;
        actor.close();
        ytClient.close();
    }

    @YTreeObject
    static class TableRow {
        public String abc_slug;
        public String project_id;
        public Map<String, String> severity_status = new HashMap<>();
        public Map<String, List<ChannelStatus>> channel_status = new HashMap<>();
    }

    @YTreeObject
    static record ChannelStatus(String type, String severity, String error, Boolean status) {
    }

    private class YtTableWriter {
        private final List<TableRow> result;
        private final CompletableFuture<Void> done = new CompletableFuture<>();
        private final YPath table;
        private final AtomicBoolean writeDone;

        public YtTableWriter(List<TableRow> resultTable) {
            table = YPath.simple(config.getYtChannelsResultTable());
            writeDone = new AtomicBoolean(false);
            result = resultTable;
        }

        public CompletableFuture<Void> write() {
            ytClient.writeTable(new WriteTable<>(table, YTreeObjectSerializerFactory.forClass(TableRow.class)))
                    .thenAccept(this::writeNext)
                    .exceptionally(ex -> {
                        handleDone(ex);
                        return null;
                    });
            return done;
        }

        private void writeNext(TableWriter<TableRow> writer) {
            if (writeDone.get()) {
                writer.close().whenComplete((unused, ex) -> handleDone(ex));
            } else {
                writer.readyEvent()
                        .thenAccept(unused -> {
                            try {
                                boolean accepted = writer.write(result);
                                if (accepted) {
                                    writeDone.set(true);
                                }
                            } catch (Exception ex) {
                                handleDone(ex);
                            }
                        })
                        .thenAcceptAsync((unused) -> writeNext(writer))
                        .exceptionally(ex -> {
                            handleDone(ex);
                            return null;
                        });
            }
        }

        private void handleDone(@Nullable Throwable ex) {
            if (ex == null) {
                done.complete(null);
            } else {
                logger.error("Error while exporting channels stats: ", ex);
                done.completeExceptionally(ex);
            }
        }
    }

    private class NotificationFetcher implements AsyncActorBody {
        private final Iterator<String> it;
        private final ConcurrentMap<String, List<TNotification>> results = new ConcurrentHashMap<>();

        public NotificationFetcher(Iterator<String> projects) {
            it = projects;
        }

        @Override
        public CompletableFuture<?> run() {
            if (!it.hasNext()) {
                return CompletableFuture.completedFuture(AsyncActorBody.DONE_MARKER);
            }
            var project = it.next();
            return fetchNotifications(project)
                    .thenApply(list -> results.put(project, list));
        }

        private CompletableFuture<List<TNotification>> fetchNotifications(String project) {
            return listProjectNotificationsRec(project, new ArrayList<>(), "");
        }

        private CompletableFuture<Pair<String, List<TNotification>>> listProjectNotificationsPage(String project, String token) {
            return notificationApi.listNotification(TListNotificationsRequest.newBuilder()
                    .setDeadlineMillis(Instant.now().plus(deadline).toEpochMilli())
                    .setProjectId(project)
                    .setPageSize(LIST_PAGE_SIZE)
                    .setPageToken(token)
                    .build())
                    .thenApply(response -> {
                        if (response.getRequestStatus() != ERequestStatusCode.OK) {
                            throw new AlertingException(response.getRequestStatus(), response.getStatusMessage());
                        }
                        return Pair.of(response.getNextPageToken(), response.getNotificationList());
                    });
        }

        private CompletableFuture<List<TNotification>> listProjectNotificationsRec(String project, List<TNotification> accumulator, String token) {
            return listProjectNotificationsPage(project, token)
                    .thenCompose(tokenAndList -> {
                        String nextToken = tokenAndList.getKey();
                        List<TNotification> page = tokenAndList.getValue();
                        accumulator.addAll(page);
                        if (nextToken.isEmpty() || page.size() < LIST_PAGE_SIZE) {
                            return completedFuture(accumulator);
                        }
                        return listProjectNotificationsRec(project, accumulator, nextToken);
                    });
        }
    }

}
