package ru.yandex.solomon.alert.stats;

import java.time.Clock;
import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.util.concurrent.MoreExecutors;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.impl.ytree.object.annotation.YTreeObject;
import ru.yandex.inside.yt.kosher.impl.ytree.object.serializers.YTreeObjectSerializerFactory;
import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.alert.api.AlertingException;
import ru.yandex.solomon.alert.client.AlertApi;
import ru.yandex.solomon.alert.cluster.broker.notification.ChannelValidationService;
import ru.yandex.solomon.alert.dao.AlertTemplateDao;
import ru.yandex.solomon.alert.dao.AlertTemplateLastVersionDao;
import ru.yandex.solomon.alert.dao.ProjectsHolder;
import ru.yandex.solomon.alert.protobuf.AlertFromTemplate;
import ru.yandex.solomon.alert.protobuf.AlertParameter;
import ru.yandex.solomon.alert.protobuf.EAlertType;
import ru.yandex.solomon.alert.protobuf.ERequestStatusCode;
import ru.yandex.solomon.alert.protobuf.TAlert;
import ru.yandex.solomon.alert.protobuf.TListAlertRequest;
import ru.yandex.solomon.alert.template.domain.AlertTemplate;
import ru.yandex.solomon.alert.template.domain.AlertTemplateId;
import ru.yandex.solomon.alert.template.domain.AlertTemplateLastVersion;
import ru.yandex.solomon.alert.template.domain.AlertTemplateParameter;
import ru.yandex.solomon.config.TimeUnitConverter;
import ru.yandex.solomon.config.protobuf.alert.ResourceMonitoringStatsConfig;
import ru.yandex.solomon.locks.DistributedLock;
import ru.yandex.solomon.locks.LockSubscriber;
import ru.yandex.solomon.locks.UnlockReason;
import ru.yandex.solomon.name.resolver.client.FindRequest;
import ru.yandex.solomon.name.resolver.client.NameResolverClient;
import ru.yandex.solomon.name.resolver.client.Resource;
import ru.yandex.solomon.secrets.SecretProvider;
import ru.yandex.solomon.selfmon.counters.AsyncMetrics;
import ru.yandex.solomon.util.actors.AsyncActorBody;
import ru.yandex.solomon.util.actors.AsyncActorRunner;
import ru.yandex.solomon.util.actors.PingActorRunner;
import ru.yandex.solomon.util.time.DurationUtils;
import ru.yandex.yt.ytclient.proxy.TableWriter;
import ru.yandex.yt.ytclient.proxy.YtClient;
import ru.yandex.yt.ytclient.proxy.request.WriteTable;
import ru.yandex.yt.ytclient.rpc.RpcCredentials;

import static java.util.concurrent.CompletableFuture.completedFuture;

/**
 * @author Alexey Trushkin
 */
@ParametersAreNonnullByDefault
public class ResourceMonitoringStatsExporter implements AutoCloseable {
    private static final Logger logger = LoggerFactory.getLogger(ResourceMonitoringStatsExporter.class);
    private static final Duration deadline = Duration.ofSeconds(30);
    private static final int ALERTING_LIST_PAGE_SIZE = 1000;

    private final DistributedLock lock;
    private final Clock clock;
    private final long initialDelayMillis;
    private final long minReindexPeriod;
    private final ProjectsHolder projectsHolder;
    private final AlertApi alertApi;
    private final AlertTemplateDao alertTemplateDao;
    private final AlertTemplateLastVersionDao alertTemplateLastVersionDao;
    private final NameResolverClient nameResolverClient;
    private final ChannelValidationService channelValidationService;
    private final YtClient ytClient;
    private final ResourceMonitoringStatsConfig config;
    private volatile long latestReindexTs;
    private volatile boolean closed;
    private final ScheduledExecutorService timer;
    private final PingActorRunner actor;
    private final AsyncMetrics metrics;

    public ResourceMonitoringStatsExporter(
            ProjectsHolder projectsHolder,
            DistributedLock lock,
            ExecutorService executor,
            ScheduledExecutorService timer,
            Clock clock,
            MetricRegistry registry,
            ResourceMonitoringStatsConfig config,
            SecretProvider secretProvider,
            AlertApi alertApi,
            AlertTemplateDao alertTemplateDao,
            AlertTemplateLastVersionDao alertTemplateLastVersionDao,
            NameResolverClient nameResolverClient,
            ChannelValidationService channelValidationService)
    {
        this.projectsHolder = projectsHolder;
        this.alertApi = alertApi;
        this.alertTemplateDao = alertTemplateDao;
        this.alertTemplateLastVersionDao = alertTemplateLastVersionDao;
        this.nameResolverClient = nameResolverClient;
        this.channelValidationService = channelValidationService;
        var intervalMillis = TimeUnitConverter.millis(config.getResourceStatusRefreshInterval());
        this.lock = lock;
        this.clock = clock;
        this.initialDelayMillis = Duration.ofMinutes(1).toMillis();
        minReindexPeriod = intervalMillis / 2;
        this.timer = timer;
        this.metrics = new AsyncMetrics(registry, "resources.monitoring_stats.exporter", Labels.of("type", "alerts"));
        this.ytClient = YtClient.builder()
                .setCluster(config.getYtCluster())
                .setRpcCredentials(new RpcCredentials("robot-solomon", secretProvider.getSecret(config.getOAuthToken()).orElseThrow()))
                .build();
        this.config = config;
        this.actor = PingActorRunner.newBuilder()
                .executor(executor)
                .timer(timer)
                .operation("Export resource monitoring stats")
                .pingInterval(Duration.ofMillis(intervalMillis))
                .backoffDelay(Duration.ofMinutes(1))
                .onPing(this::act)
                .build();
        acquireLock();
    }

    public CompletableFuture<Void> exportStats() {
        var iterator = projectsHolder.getProjects().stream()
                .filter(project -> !config.getSkipYasm() || !project.startsWith("yasm_"))
                .iterator();
        var actorBody = new AlertFetcher(iterator);
        var runner = new AsyncActorRunner(actorBody, MoreExecutors.directExecutor(), 200);
        return runner.start()
                .thenApply(unused -> actorBody.results.values())
                .thenCompose(alertLists -> nameResolverClient.getShardIds()
                        .thenCompose(shardIds -> {
                            var resourceFetcher = new ResourceFetcher(shardIds.ids().iterator());
                            return new AsyncActorRunner(resourceFetcher, MoreExecutors.directExecutor(), 200).start()
                                    .thenCompose(unused -> alertTemplateDao.getAll()
                                            .thenCompose(alertTemplates -> alertTemplateLastVersionDao.getAll().thenCompose(versions -> {
                                                var alerts = alertLists.stream()
                                                        .flatMap(Collection::stream)
                                                        .collect(Collectors.toList());
                                                return CompletableFutures.safeCall(() -> prepareResults(resourceFetcher.results, alerts, alertTemplates, versions));
                                            }))
                                            .thenCompose(tableRows -> new YtTableWriter(tableRows).write()));
                        }));
    }

    private CompletableFuture<List<TableRow>> prepareResults(
            Map<String, List<Resource>> resourcesMap,
            List<TAlert> alerts,
            List<AlertTemplate> alertTemplates,
            List<AlertTemplateLastVersion> versions)
    {
        CompletableFuture<Void> done = CompletableFuture.completedFuture(null);
        var templates = alertTemplates.stream()
                .collect(Collectors.toMap(AlertTemplate::getCompositeId, Function.identity()));
        Map<String, List<AlertTemplateLastVersion>> spDefaultTemplates = new HashMap<>();
        for (AlertTemplateLastVersion version : versions) {
            if (templates.get(new AlertTemplateId(version.id(), version.templateVersionTag())).isDefaultTemplate()) {
                var list = spDefaultTemplates.computeIfAbsent(version.serviceProviderId(), s -> new ArrayList<>());
                list.add(version);
            }
        }
        Map<String, Map<Map<String, String>, Map<String, List<TAlert>>>> spResourceAlerts = new HashMap<>();
        for (TAlert alert : alerts) {
            final AlertFromTemplate alertFromTemplate = alert.getAlertFromTemplate();
            var template = templates.get(new AlertTemplateId(alertFromTemplate.getTemplateId(), alertFromTemplate.getTemplateVersionTag()));
            var resources = spResourceAlerts.computeIfAbsent(template.getServiceProviderId(), s -> new HashMap<>());
            var templatesCreated = resources.computeIfAbsent(toResourceId(alertFromTemplate, template), stringStringMap -> new HashMap<>());
            var list = templatesCreated.getOrDefault(template.getId(), new ArrayList<>());
            list.add(alert);
            templatesCreated.put(template.getId(), list);
        }

        List<TableRow> rows = new ArrayList<>();
        for (List<Resource> rowList : resourcesMap.values()) {
            for (Resource resource : rowList) {
                var tableRow = TableRow.of(resource);
                rows.add(tableRow);
                var templatesNeeded = spDefaultTemplates.get(tableRow.service_provider_id);
                if (templatesNeeded == null || templatesNeeded.isEmpty()) {
                    continue;
                }
                var resources = spResourceAlerts.get(tableRow.service_provider_id);
                if (resources == null || resources.isEmpty()) {
                    fillEmptyTemplate(tableRow, templatesNeeded, templates);
                    continue;
                }
                var createdAlerts = resources.get(tableRow.resource_id);
                if (createdAlerts == null || createdAlerts.isEmpty()) {
                    fillEmptyTemplate(tableRow, templatesNeeded, templates);
                    continue;
                }
                var stats = new HashMap<String, String>();
                var alertChannelStats = new HashMap<String, String>();
                for (AlertTemplateLastVersion version : templatesNeeded) {
                    var template = templates.get(new AlertTemplateId(version.id(), version.templateVersionTag()));
                    if (suitableTemplate(tableRow.resource_type, template)) {
                        var currentAlerts = createdAlerts.getOrDefault(version.id(), List.of());
                        stats.put(text(version), String.valueOf(!currentAlerts.isEmpty()));
                        for (var alert : currentAlerts) {
                            if ("true".equals(alertChannelStats.get(text(version)))) {
                                continue;
                            }
                            done = done.thenCompose(ignore -> validateChannels(alert, template, resource)
                                    .thenAccept(aBoolean -> alertChannelStats.put(text(version), String.valueOf(aBoolean))));
                        }
                    }
                }
                tableRow.monitoring_stats.put("alerts_status", stats);
                tableRow.monitoring_stats.put("alerts_channel_status", alertChannelStats);
            }
        }
        return done
                .thenApply(unused -> rows);

    }

    private CompletableFuture<Boolean> validateChannels(TAlert alert, AlertTemplate template, Resource resource) {
        return channelValidationService.validateChannelsForSeverity(alert.getNotificationChannelIdsList(), template.getSeverity(), resource.severity);
    }

    private Map<String, String> toResourceId(AlertFromTemplate alertFromTemplate, AlertTemplate template) {
        var actualParameters = template.getParameters().stream()
            .map(AlertTemplateParameter::getName)
            .collect(Collectors.toSet());
        HashMap<String, String> resourceId = new HashMap<>();
        for (AlertParameter p : alertFromTemplate.getAlertParametersList()) {
            switch (p.getParameterCase()) {
                case DOUBLE_PARAMETER_VALUE -> {
                    if (actualParameters.contains(p.getDoubleParameterValue().getName())) {
                        resourceId.put(p.getDoubleParameterValue().getName(), String.valueOf(p.getDoubleParameterValue().getValue()));
                    }
                }
                case INTEGER_PARAMETER_VALUE -> {
                    if (actualParameters.contains(p.getIntegerParameterValue().getName())) {
                        resourceId.put(p.getIntegerParameterValue().getName(), String.valueOf(p.getIntegerParameterValue().getValue()));
                    }
                }
                case TEXT_PARAMETER_VALUE -> {
                    if (actualParameters.contains(p.getTextParameterValue().getName())) {
                        resourceId.put(p.getTextParameterValue().getName(), p.getTextParameterValue().getValue());
                    }
                }
                case TEXT_LIST_PARAMETER_VALUE -> {
                    if (actualParameters.contains(p.getTextListParameterValue().getName())) {
                        resourceId.put(p.getTextListParameterValue().getName(), p.getTextListParameterValue().getValuesList().toString());
                    }
                }
                case LABEL_LIST_PARAMETER_VALUE -> {
                    if (actualParameters.contains(p.getLabelListParameterValue().getName())) {
                        resourceId.put(p.getLabelListParameterValue().getName(), p.getLabelListParameterValue().getValuesList().toString());
                    }
                }
            }
        }
        return resourceId;
    }

    private void fillEmptyTemplate(TableRow tableRow, List<AlertTemplateLastVersion> templatesNeeded, Map<AlertTemplateId, AlertTemplate> templates) {
        var stats = new HashMap<String, String>();
        for (AlertTemplateLastVersion version : templatesNeeded) {
            var template = templates.get(new AlertTemplateId(version.id(), version.templateVersionTag()));
            if (suitableTemplate(tableRow.resource_type, template)) {
                stats.put(text(version), "false");
            }
        }
        tableRow.monitoring_stats.put("alerts_status", stats);
        tableRow.monitoring_stats.put("alerts_channel_status", stats);
    }

    private boolean suitableTemplate(String resourceType, @Nullable AlertTemplate template) {
        if (template == null || StringUtils.isEmpty(resourceType)) {
            return true;
        }
        var templateResourceType = new HashSet<>(Arrays.asList(template.getLabels().getOrDefault("resourceType", "all").split("\\|")));
        if (templateResourceType.contains("all")) {
            return true;
        }
        return templateResourceType.contains(resourceType);
    }

    private String text(AlertTemplateLastVersion version) {
        return version.name() + "(" + version.id() + ")";
    }

    public CompletableFuture<Void> act(int attempt) {
        if (closed) {
            return completedFuture(null);
        }

        if (!lock.isLockedByMe()) {
            return completedFuture(null);
        }
        long reindexTs = clock.millis();
        if (minReindexPeriod > reindexTs - latestReindexTs) {
            return completedFuture(null);
        }

        var future = exportStats()
                .thenAccept(ignore -> latestReindexTs = reindexTs);
        metrics.forFuture(future);
        return future;
    }

    public void schedule() {
        long delay = DurationUtils.randomize(initialDelayMillis);
        timer.schedule(actor::forcePing, delay, TimeUnit.MILLISECONDS);
    }

    private void acquireLock() {
        if (closed) {
            return;
        }

        lock.acquireLock(new LockSubscriber() {
            @Override
            public boolean isCanceled() {
                return closed;
            }

            @Override
            public void onLock(long seqNo) {
                logger.info("Acquire ResourceMonitoringStatsExporter lock, seqNo {}", seqNo);
                schedule();
            }

            @Override
            public void onUnlock(UnlockReason reason) {
                logger.info("Loose ResourceMonitoringStatsExporter lock by reason: {}", reason);
                acquireLock();
            }
        }, 5, TimeUnit.MINUTES);
    }

    @Override
    public void close() {
        closed = true;
        actor.close();
        ytClient.close();
    }

    @YTreeObject
    static class TableRow {
        public String abc_slug;
        public Integer abc_id = -1;
        public Map<String, String> resource_id;
        public String service_provider_id;
        public String resource_type;
        public String responsible;
        public String environment;
        public String severity;
        public Map<String, Map<String, String>> monitoring_stats;

        public static TableRow of(Resource resource) {
            TableRow r = new TableRow();
            r.abc_slug = resource.cloudId;
            r.resource_id = resource.resourceComplexId;
            r.service_provider_id = resource.service;
            r.resource_type = resource.type;
            r.responsible = resource.responsible;
            r.environment = resource.environment;
            r.severity = resource.severity == Resource.Severity.UNKNOWN
                    ? Resource.Severity.HIGHLY_CRITICAL.name()
                    : resource.severity.name();
            r.fillNulls();
            return r;
        }

        private void fillNulls() {
            if (monitoring_stats == null) {
                monitoring_stats = new HashMap<>();
            }
            if (environment == null) {
                environment = "";
            }
            if (resource_type == null) {
                resource_type = "";
            }
            if (abc_slug == null) {
                abc_slug = "";
            }
            if (responsible == null) {
                responsible = "";
            }
            if (abc_id == null) {
                abc_id = -1;
            }
        }
    }

    private class YtTableWriter {
        private final List<TableRow> result;
        private final CompletableFuture<Void> done = new CompletableFuture<>();
        private final YPath table;
        private final AtomicBoolean writeDone;

        public YtTableWriter(List<TableRow> resultTable) {
            table = YPath.simple(config.getYtResultTable());
            writeDone = new AtomicBoolean(false);
            result = resultTable;
        }

        public CompletableFuture<Void> write() {
            ytClient.writeTable(new WriteTable<>(table, YTreeObjectSerializerFactory.forClass(TableRow.class)))
                    .thenAccept(this::writeNext)
                    .exceptionally(ex -> {
                        handleDone(ex);
                        return null;
                    });
            return done;
        }

        private void writeNext(TableWriter<TableRow> writer) {
            if (writeDone.get()) {
                writer.close().whenComplete((unused, ex) -> handleDone(ex));
            } else {
                writer.readyEvent()
                        .thenAccept(unused -> {
                            try {
                                boolean accepted = writer.write(result);
                                if (accepted) {
                                    writeDone.set(true);
                                }
                            } catch (Exception ex) {
                                handleDone(ex);
                            }
                        })
                        .thenAcceptAsync((unused) -> writeNext(writer))
                        .exceptionally(ex -> {
                            handleDone(ex);
                            return null;
                        });
            }
        }

        private void handleDone(@Nullable Throwable ex) {
            if (ex == null) {
                done.complete(null);
            } else {
                logger.error("Error while exporting resources stats: ", ex);
                done.completeExceptionally(ex);
            }
        }
    }

    private class AlertFetcher implements AsyncActorBody {
        private final Iterator<String> it;
        private final ConcurrentMap<String, List<TAlert>> results = new ConcurrentHashMap<>();

        public AlertFetcher(Iterator<String> projects) {
            it = projects;
        }

        @Override
        public CompletableFuture<?> run() {
            if (!it.hasNext()) {
                return CompletableFuture.completedFuture(AsyncActorBody.DONE_MARKER);
            }
            var project = it.next();
            return fetchAlerts(project)
                    .thenApply(tAlerts -> results.put(project, tAlerts));
        }

        private CompletableFuture<List<TAlert>> fetchAlerts(String project) {
            return listProjectAlertsRec(project, new ArrayList<>(), "");
        }

        private CompletableFuture<Pair<String, List<TAlert>>> listProjectAlertsPage(String project, String token) {
            return alertApi.listAlerts(TListAlertRequest.newBuilder()
                    .setDeadlineMillis(Instant.now().plus(deadline).toEpochMilli())
                    .setProjectId(project)
                    .addFilterByType(EAlertType.FROM_TEMPLATE)
                    .setFullResultModel(true)
                    .setPageSize(ALERTING_LIST_PAGE_SIZE)
                    .setPageToken(token)
                    .build())
                    .thenApply(response -> {
                        if (response.getRequestStatus() != ERequestStatusCode.OK) {
                            throw new AlertingException(response.getRequestStatus(), response.getStatusMessage());
                        }
                        return Pair.of(response.getNextPageToken(), response.getAlertList().getAlertsList());
                    });
        }

        private CompletableFuture<List<TAlert>> listProjectAlertsRec(String project, List<TAlert> accumulator, String token) {
            return listProjectAlertsPage(project, token)
                    .thenCompose(tokenAndList -> {
                        String nextToken = tokenAndList.getKey();
                        List<TAlert> page = tokenAndList.getValue();
                        accumulator.addAll(page);
                        if (nextToken.isEmpty() || page.size() < ALERTING_LIST_PAGE_SIZE) {
                            return completedFuture(accumulator);
                        }
                        return listProjectAlertsRec(project, accumulator, nextToken);
                    });
        }
    }

    private class ResourceFetcher implements AsyncActorBody {
        private final Iterator<String> it;
        private final ConcurrentMap<String, List<Resource>> results = new ConcurrentHashMap<>();

        public ResourceFetcher(Iterator<String> shards) {
            it = shards;
        }

        @Override
        public CompletableFuture<?> run() {
            if (!it.hasNext()) {
                return CompletableFuture.completedFuture(AsyncActorBody.DONE_MARKER);
            }
            var shardId = it.next();
            return fetchResources(shardId)
                    .thenApply(tAlerts -> results.put(shardId, tAlerts));
        }

        private CompletableFuture<List<Resource>> fetchResources(String shardId) {
            final FindRequest findRequest = FindRequest.newBuilder()
                    .cloudId(shardId)
                    .filterDeleted(true)
                    .build();
            return nameResolverClient.find(findRequest)
                    .thenApply(findResponse -> findResponse.resources);
        }
    }

}
