package ru.yandex.solomon.alert.tasks;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;

import ru.yandex.alerting.api.task.PublishAlertTemplateParams;
import ru.yandex.solomon.alert.client.AlertApi;
import ru.yandex.solomon.alert.protobuf.UpdateAlertTemplateVersionRequest;
import ru.yandex.solomon.scheduler.ExecutionContext;
import ru.yandex.solomon.util.actors.AsyncActorBody;
import ru.yandex.solomon.util.actors.AsyncActorRunner;
import ru.yandex.solomon.util.future.RetryCompletableFuture;
import ru.yandex.solomon.util.future.RetryConfig;

/**
 * @author Alexey Trushkin
 */
class PublishAlertTemplaForProjectTask {
    private static final int ALERT_INFLIGHT = 10;
    private static final RetryConfig RETRY_CONFIG = RetryConfig.DEFAULT
            .withNumRetries(5)
            .withDelay(TimeUnit.SECONDS.toMillis(1))
            .withMaxDelay(TimeUnit.MINUTES.toMillis(1));
    private final String project;
    private final String templateId;
    private final String templateVersionTag;
    private final Executor executor;
    private final AlertApi alertApi;
    private final ExecutionContext context;
    private final AtomicInteger updatedAlerts = new AtomicInteger(0);
    private volatile boolean continueUpdate = true;

    public PublishAlertTemplaForProjectTask(
            String project,
            PublishAlertTemplateParams params,
            Executor executor,
            AlertApi alertApi,
            ExecutionContext context)
    {
        this.project = project;
        this.templateId = params.getTemplateId();
        this.templateVersionTag = params.getTemplateVersionTag();
        this.executor = executor;
        this.alertApi = alertApi;
        this.context = context;
    }

    public CompletableFuture<Integer> start() {
        AsyncActorBody body = () -> {
            while (true) {
                return RetryCompletableFuture.runWithRetries(() -> {
                    if (!continueUpdate || context.isDone()) {
                        return CompletableFuture.completedFuture(AsyncActorBody.DONE_MARKER);
                    }
                    var request = UpdateAlertTemplateVersionRequest.newBuilder()
                            .setTemplateId(templateId)
                            .setTemplateVersionTag(templateVersionTag)
                            .setUpdateCount(ALERT_INFLIGHT)
                            .setProjectId(project)
                            .build();
                    return alertApi.updateAlertTemplateVersion(request)
                            .thenApply(response -> {
                                continueUpdate = response.getUpdated() == ALERT_INFLIGHT;
                                updatedAlerts.addAndGet(response.getUpdated());
                                return response;
                            });
                }, RETRY_CONFIG);
            }
        };
        var runner = new AsyncActorRunner(body, executor, 1);
        return runner.start()
                .thenApply(unused -> updatedAlerts.get());
    }
}
