package ru.yandex.solomon.alert.tasks;

import java.util.Iterator;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.Collectors;

import ru.yandex.alerting.api.task.PublishAlertTemplateParams;
import ru.yandex.alerting.api.task.PublishAlertTemplateProgress;
import ru.yandex.alerting.api.task.PublishAlertTemplateResult;
import ru.yandex.solomon.alert.client.AlertApi;
import ru.yandex.solomon.alert.dao.ProjectsHolder;
import ru.yandex.solomon.scheduler.ExecutionContext;
import ru.yandex.solomon.util.actors.AsyncActorBody;
import ru.yandex.solomon.util.actors.AsyncActorRunner;

/**
 * @author Alexey Trushkin
 */
class PublishAlertTemplateTask {

    private static final int PROJECT_INFLIGHT = 1;
    private final ExecutionContext context;
    private final AlertApi alertApi;
    private final ProjectsHolder projectsHolder;
    private final Executor executor;
    private final int totalProjects;
    private volatile boolean closed;
    private final AtomicReference<PublishAlertTemplateProgress> progressAtomicReference;

    public PublishAlertTemplateTask(
            ExecutionContext context,
            AlertApi alertApi,
            ProjectsHolder projectsHolder,
            Executor executor)
    {
        this.context = context;
        this.alertApi = alertApi;
        this.projectsHolder = projectsHolder;
        this.executor = executor;
        totalProjects = projectsHolder.getProjects().size();
        progressAtomicReference = new AtomicReference<>(PublishAlertTemplateProto.progress(context.task().progress()));
    }

    public CompletableFuture<PublishAlertTemplateResult> start() {
        PublishAlertTemplateParams params = PublishAlertTemplateProto.params(context.task().params());
        PublishAlertTemplateProgress progress = progressAtomicReference.get();
        String lastProjectId = progress.getLastProjectId();
        List<String> projects = projectsHolder.getProjects().stream()
                .filter(s -> lastProjectId.isEmpty() || s.compareTo(lastProjectId) > 0)
                .sorted()
                .collect(Collectors.toList());

        Iterator<String> projectsIterator = projects.iterator();
        AsyncActorBody body = () -> {
            while (true) {
                if (!projectsIterator.hasNext() || context.isDone()) {
                    return CompletableFuture.completedFuture(AsyncActorBody.DONE_MARKER);
                }
                if (closed) {
                    throw new IllegalStateException("PublishAlertTemplateTask is closed, id " + context.task().id());
                }
                var project = projectsIterator.next();
                var process = new PublishAlertTemplaForProjectTask(project, params, executor, alertApi, context);
                return process.start()
                        .thenCompose(updatedAlerts -> updateProgress(updatedAlerts, project));
            }
        };
        var runner = new AsyncActorRunner(body, executor, PROJECT_INFLIGHT);
        return runner.start()
                .thenCompose(updated -> complete(context));
    }

    private CompletableFuture<?> updateProgress(Integer updatedAlerts, String project) {
        PublishAlertTemplateProgress progressLast = progressAtomicReference.get();
        PublishAlertTemplateProgress progress = progressLast.toBuilder()
                .setLastProjectId(project)
                .setUpdatedAlerts(progressLast.getUpdatedAlerts() + updatedAlerts)
                .setUpdatedProjects(progressLast.getUpdatedProjects() + 1)
                .setTotalProjects(totalProjects)
                .build();
        progressAtomicReference.set(progress);
        return context.progress(progress);
    }


    private CompletableFuture<PublishAlertTemplateResult> complete(ExecutionContext context) {
        var result = PublishAlertTemplateResult.newBuilder()
                .setUpdatedAlerts(progressAtomicReference.get().getUpdatedAlerts())
                .build();
        return context.complete(result)
                .thenApply(o -> result);
    }


    public PublishAlertTemplateProgress getProgress() {
        return progressAtomicReference.get();
    }

    public void close() {
        closed = true;
    }
}
