package ru.yandex.solomon.alert.tasks;

import java.time.Instant;
import java.util.List;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.Executor;
import java.util.concurrent.TimeUnit;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.protobuf.Any;
import com.google.protobuf.Descriptors;
import com.google.protobuf.TextFormat;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.alerting.api.task.PublishAlertTemplateParams;
import ru.yandex.alerting.api.task.PublishAlertTemplateProgress;
import ru.yandex.alerting.api.task.PublishAlertTemplateResult;
import ru.yandex.solomon.alert.client.AlertApi;
import ru.yandex.solomon.alert.dao.ProjectsHolder;
import ru.yandex.solomon.scheduler.ExecutionContext;
import ru.yandex.solomon.scheduler.Permit;
import ru.yandex.solomon.scheduler.PermitLimiter;
import ru.yandex.solomon.scheduler.Task;
import ru.yandex.solomon.scheduler.TaskHandler;
import ru.yandex.solomon.util.time.DurationUtils;

/**
 * @author Alexey Trushkin
 */
@ParametersAreNonnullByDefault
public class PublishAlertTemplateTaskHandler implements TaskHandler {
    private static final Logger logger = LoggerFactory.getLogger(PublishAlertTemplateTaskHandler.class);
    private static final String TYPE = "publish_alert_template";

    private final PermitLimiter permitLimiter = new PermitLimiter(2);
    private final AlertApi alertApi;
    private final ProjectsHolder projectsHolder;
    private final Executor executor;
    // some state for ui
    private final ConcurrentMap<String, PublishAlertTemplateTask> running = new ConcurrentHashMap<>();

    public PublishAlertTemplateTaskHandler(
            AlertApi alertApi,
            ProjectsHolder projectsHolder,
            Executor executor)
    {
        this.alertApi = alertApi;
        this.projectsHolder = projectsHolder;
        this.executor = executor;
    }

    @Override
    public String type() {
        return TYPE;
    }

    @Override
    @Nullable
    public Permit acquire(String id, Any params) {
        if (projectsHolder.getProjects().isEmpty()) {
            return null;
        }
        if (running.containsKey(id)) {
            return null;
        }
        return permitLimiter.acquire();
    }

    @Override
    public void execute(ExecutionContext context) {
        try {
            PublishAlertTemplateTask task = new PublishAlertTemplateTask(context, alertApi, projectsHolder, executor);
            running.put(context.task().id(), task);
            task.start()
                    .whenComplete((result, throwable) -> {
                        running.remove(context.task().id(), task);
                        if (throwable != null) {
                            reschedule(throwable, context, task.getProgress());
                        } else {
                            logger.info("{} completed, result ({})", logPrefix(context), TextFormat.shortDebugString(result));
                        }
                    });
        } catch (Throwable t) {
            reschedule(t, context);
        }
    }

    private CompletableFuture<?> reschedule(Throwable t, ExecutionContext context) {
        return reschedule(t, context, PublishAlertTemplateProto.progress(context.task().progress()));
    }

    private CompletableFuture<?> reschedule(Throwable t, ExecutionContext context, PublishAlertTemplateProgress progress) {
        logger.error("{} something went wrong", logPrefix(context), t);
        long executeAt = System.currentTimeMillis() + DurationUtils.randomize(TimeUnit.MINUTES.toMillis(40));
        return context.reschedule(executeAt, progress)
                .whenComplete((o, throwable) -> {
                    logger.info("{} rescheduled on {}, latest progress ({})", logPrefix(context), Instant.ofEpochMilli(executeAt), TextFormat.shortDebugString(progress));
                });
    }

    @Override
    public List<Descriptors.Descriptor> descriptors() {
        return List.of(PublishAlertTemplateParams.getDescriptor(), PublishAlertTemplateResult.getDescriptor());
    }

    private String logPrefix(ExecutionContext context) {
        return TYPE + "(task_id: \"" + context.task().id() + "\" " + TextFormat.shortDebugString(context.task().params()) + ")";
    }

    public void close() {
        running.values().forEach(PublishAlertTemplateTask::close);
    }

    public static Task task(String templateId, String templateVersionTag, String taskId) {
        var params = Any.pack(PublishAlertTemplateParams.newBuilder()
                .setTemplateId(templateId)
                .setTemplateVersionTag(templateVersionTag)
                .build());

        long executeAt = System.currentTimeMillis() + DurationUtils.randomize(TimeUnit.SECONDS.toMillis(20));

        return Task.newBuilder()
                .setId(taskId)
                .setType(TYPE)
                .setExecuteAt(executeAt)
                .setParams(params)
                .build();
    }

    public static String taskId() {
        return UUID.randomUUID().toString();
    }
}
