package ru.yandex.solomon.alert.telegram;

import java.io.IOException;
import java.net.URI;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.time.Duration;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.concurrent.TimeUnit;

import javax.annotation.Nullable;

import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ObjectNode;
import com.google.common.annotations.VisibleForTesting;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.commune.json.jackson.StringEnumModule;
import ru.yandex.misc.lang.Validate;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.alert.telegram.dto.InlineKeyboard;
import ru.yandex.solomon.alert.telegram.dto.ParseMode;
import ru.yandex.solomon.alert.telegram.dto.TelegramMessage;
import ru.yandex.solomon.alert.telegram.dto.TelegramSendMessage;
import ru.yandex.solomon.alert.telegram.dto.TelegramUpdate;
import ru.yandex.solomon.selfmon.counters.AsyncMetrics;
import ru.yandex.solomon.staffOnly.annotations.ManagerMethod;


/**
 * @author alexlovkov
 **/
public class DefaultTelegramClient implements TelegramClient {
    private static final Logger logger = LoggerFactory.getLogger(DefaultTelegramClient.class);

    private static final int DEFAULT_TIMEOUT_SECONDS = 15;

    private static final ObjectMapper mapper;
    static {
        mapper = new ObjectMapper();
        mapper.registerModule(new StringEnumModule());
    }

    private final Executor executor;
    private volatile HttpClient httpClient;
    private final Endpoint endpointGetMe;
    private final Endpoint endpointGetUpdates;
    private final Endpoint endpointSendMessage;
    private final Endpoint endpointSendPhoto;
    private final Endpoint endpointAnswerCallbackQuery;
    private final Endpoint endpointEditMessageReplyMarkup;
    private final Endpoint endpointEditMessageText;

    public DefaultTelegramClient(String url, String botToken, Executor executor) {
        this(url, botToken, MetricRegistry.root(), executor);
    }

    private HttpClient makeHttpClient() {
        HttpClient.Builder builder = HttpClient.newBuilder()
                .followRedirects(HttpClient.Redirect.NEVER)
                .connectTimeout(Duration.ofSeconds(5));
        if (executor != null) {
            builder.executor(executor);
        }
        return builder.build();
    }

    @ManagerMethod
    public void recreateHttpClient() {
        this.httpClient = makeHttpClient();
    }

    public DefaultTelegramClient(String url, String botToken, MetricRegistry reg, @Nullable Executor executor) {
        this.executor = executor;
        this.httpClient = makeHttpClient();
        final String prefixUri = url + "/bot" + botToken;
        this.endpointGetMe = new Endpoint(reg, prefixUri, "/getMe");
        this.endpointGetUpdates = new Endpoint(reg, prefixUri, "/getUpdates");
        this.endpointSendMessage = new Endpoint(reg, prefixUri, "/sendMessage");
        this.endpointSendPhoto = new Endpoint(reg, prefixUri, "/sendPhoto");
        this.endpointAnswerCallbackQuery = new Endpoint(reg, prefixUri, "/answerCallbackQuery");
        this.endpointEditMessageReplyMarkup = new Endpoint(reg, prefixUri, "/editMessageReplyMarkup");
        this.endpointEditMessageText = new Endpoint(reg, prefixUri, "/editMessageText");
    }

    @Override
    public CompletableFuture<String> getMe() {
        var request = HttpRequest.newBuilder()
            .timeout(Duration.ofSeconds(DEFAULT_TIMEOUT_SECONDS));

        return endpointGetMe.send(request)
            .thenApply(HttpResponse::body);
    }

    @Override
    public CompletableFuture<List<TelegramUpdate>> getUpdates(long offset, int limit, int timeoutSeconds) {
        ObjectNode body = mapper.createObjectNode();
        body.put("offset", Long.toString(offset));
        body.put("limit", limit);
        body.put("timeout", timeoutSeconds);

        var request = HttpRequest.newBuilder()
            .header("Content-Type", "application/json")
            .POST(HttpRequest.BodyPublishers.ofString(body.toString()))
            .timeout(Duration.ofSeconds(DEFAULT_TIMEOUT_SECONDS));

        return endpointGetUpdates.send(request)
            .thenApply(response -> {
                if (response.statusCode() == 200) {
                    return parseUpdatesResponse(response.body());
                }
                throw new IllegalStateException(
                    "Got " + response.statusCode() + " on getUpdate. Response: " + response.body());
            });
    }

    @Override
    public CompletableFuture<TelegramSendMessage> sendMessage(
        long chatId,
        String text,
        ParseMode parseMode,
        @Nullable InlineKeyboard inlineKeyboard,
        long replyMessageId)
    {
        ObjectNode body = mapper.createObjectNode();
        body.put("text", text);
        body.put("chat_id", chatId);
        body.put("disable_web_page_preview", true);
        if (parseMode != ParseMode.PLAIN && parseMode.value().isPresent()) {
            body.put("parse_mode", parseMode.value().get());
        }
        if (inlineKeyboard != null) {
            try {
                String keyboardString = mapper.writeValueAsString(inlineKeyboard);
                body.put("reply_markup", keyboardString);
            } catch (JsonProcessingException e) {
                return CompletableFuture.failedFuture(e);
            }
        }
        if (replyMessageId != 0) {
            body.put("reply_to_message_id", replyMessageId);
        }

        var request = HttpRequest.newBuilder()
            .header("Content-Type", "application/json")
            .timeout(Duration.ofSeconds(DEFAULT_TIMEOUT_SECONDS))
            .POST(HttpRequest.BodyPublishers.ofString(body.toString()));

        return endpointSendMessage.send(request)
            .thenApply(this::handleTelegramMessage);
    }

    @Override
    public CompletableFuture<TelegramSendMessage> forceReply(long chatId, String text, ParseMode parseMode, long replyMessageId) {
        ObjectNode body = mapper.createObjectNode();
        body.put("text", text);
        body.put("chat_id", chatId);
        body.put("disable_web_page_preview", true);
        if (parseMode != ParseMode.PLAIN && parseMode.value().isPresent()) {
            body.put("parse_mode", parseMode.value().get());
        }
        if (replyMessageId != 0) {
            body.put("reply_to_message_id", replyMessageId);
        }
        ObjectNode forceReply = body.putObject("reply_markup");
        forceReply.put("force_reply", true);

        var request = HttpRequest.newBuilder()
                .header("Content-Type", "application/json")
                .timeout(Duration.ofSeconds(DEFAULT_TIMEOUT_SECONDS))
                .POST(HttpRequest.BodyPublishers.ofString(body.toString()));

        return endpointSendMessage.send(request)
                .thenApply(this::handleTelegramMessage);
    }

    @Override
    public CompletableFuture<TelegramSendMessage> sendPhoto(
        long chatId,
        byte[] photo,
        @Nullable String captionText,
        @Nullable ParseMode parseMode,
        long replyMessageId,
        @Nullable InlineKeyboard inlineKeyboard)
    {
        MultiPartBodyPublisher publisher = new MultiPartBodyPublisher()
            .addPart("chat_id", String.valueOf(chatId))
            .addPart("photo", photo);
        if (parseMode != null && parseMode != ParseMode.PLAIN && parseMode.value().isPresent()) {
            publisher.addPart("parse_mode", parseMode.value().get());
        }
        if (captionText != null) {
            publisher.addPart("caption", captionText);
        }
        if (replyMessageId != 0) {
            publisher.addPart("reply_to_message_id", String.valueOf(replyMessageId));
        }
        if (inlineKeyboard != null) {
            try {
                String keyboardString = mapper.writeValueAsString(inlineKeyboard);
                publisher.addPart("reply_markup", keyboardString);
            } catch (JsonProcessingException e) {
                return CompletableFuture.failedFuture(e);
            }
        }

        var request = HttpRequest.newBuilder()
            .header("Content-Type", "multipart/form-data; boundary=" + publisher.getBoundary())
            .timeout(Duration.ofSeconds(DEFAULT_TIMEOUT_SECONDS))
            .POST(publisher.build());

        return endpointSendPhoto.send(request)
            .thenApply(this::handleTelegramMessage);
    }

    @Override
    public CompletableFuture<TelegramSendMessage> answerCallbackQuery(
        String callbackId,
        @Nullable String text,
        @Nullable Boolean showAlert,
        @Nullable String url,
        @Nullable Integer cacheTimeSeconds)
    {
        ObjectNode body = mapper.createObjectNode();
        body.put("callback_query_id", callbackId);
        if (text != null) {
            body.put("text", text);
        }
        if (showAlert != null) {
            body.put("show_alert", showAlert);
        }
        if (url != null) {
            body.put("url", url);
        }
        if (cacheTimeSeconds != null) {
            body.put("cache_time", cacheTimeSeconds);
        }

        var request = HttpRequest.newBuilder()
            .header("Content-Type", "application/json")
            .timeout(Duration.ofSeconds(DEFAULT_TIMEOUT_SECONDS))
            .POST(HttpRequest.BodyPublishers.ofString(body.toString()));

        return endpointAnswerCallbackQuery.send(request)
            .thenApply(this::handleTelegramMessage);
    }

    @Override
    public CompletableFuture<TelegramSendMessage> editMessageReplyMarkup(
            long chatId,
            long messageId,
            InlineKeyboard inlineKeyboard)
    {
        ObjectNode body = mapper.createObjectNode();
        body.put("chat_id", chatId);
        body.put("message_id", messageId);
        if (inlineKeyboard == null) {
            inlineKeyboard = InlineKeyboard.EMPTY;
        }
        try {
            String keyboardString = mapper.writeValueAsString(inlineKeyboard);
            body.put("reply_markup", keyboardString);
        } catch (JsonProcessingException e) {
            return CompletableFuture.failedFuture(e);
        }

        var request = HttpRequest.newBuilder()
                .header("Content-Type", "application/json")
                .timeout(Duration.ofSeconds(DEFAULT_TIMEOUT_SECONDS))
                .POST(HttpRequest.BodyPublishers.ofString(body.toString()));

        return endpointEditMessageReplyMarkup.send(request)
                .thenApply(this::handleTelegramMessage);
    }

    @Override
    public CompletableFuture<TelegramSendMessage> editMessageText(
            long chatId,
            long messageId,
            String text,
            @Nullable ParseMode parseMode)
    {
        ObjectNode body = mapper.createObjectNode();
        body.put("chat_id", chatId);
        body.put("message_id", messageId);
        if (parseMode != null && parseMode != ParseMode.PLAIN && parseMode.value().isPresent()) {
            body.put("parse_mode", parseMode.value().get());
        }
        body.put("text", text);

        var request = HttpRequest.newBuilder()
                .header("Content-Type", "application/json")
                .timeout(Duration.ofSeconds(DEFAULT_TIMEOUT_SECONDS))
                .POST(HttpRequest.BodyPublishers.ofString(body.toString()));

        return endpointEditMessageText.send(request)
                .thenApply(this::handleTelegramMessage);
    }

    private TelegramSendMessage handleTelegramMessage(HttpResponse<String> response) {
        SendResponse telegramMessage = parseSendMessage(response.body());
        if (telegramMessage.isOk()) {
            return new TelegramSendMessage(response.statusCode(), telegramMessage.getResult().getId());
        }
        return new TelegramSendMessage(response.statusCode(), telegramMessage.getDescription());
    }

    @Override
    public void close() {
    }

    private static class TelegramInnerResponse {

        private boolean ok;
        private String description;

        public boolean isOk() {
            return ok;
        }

        @JsonProperty("ok")
        public void setOk(boolean ok) {
            this.ok = ok;
        }

        @JsonProperty("description")
        public void setDescription(String description) {
            this.description = description;
        }

        public String getDescription() {
            return description;
        }
    }

    @JsonIgnoreProperties(ignoreUnknown = true)
    private static final class UpdatesResponse extends TelegramInnerResponse {

        private List<TelegramUpdate> result;

        public List<TelegramUpdate> getResult() {
            return result;
        }

        @JsonProperty("result")
        public void setResult(List<TelegramUpdate> result) {
            this.result = result;
        }
    }

    @JsonIgnoreProperties(ignoreUnknown = true)
    private static final class SendResponse extends TelegramInnerResponse {

        private TelegramMessage result;

        public TelegramMessage getResult() {
            return result;
        }

        @JsonProperty("result")
        public void setResult(TelegramMessage result) {
            this.result = result;
        }
    }

    @VisibleForTesting
    @JsonIgnoreProperties(ignoreUnknown = true)
    public static List<TelegramUpdate> parseUpdatesResponse(String response) {
        try {
            UpdatesResponse updatesResponse =
                mapper.readValue(response, UpdatesResponse.class);
            Validate.isTrue(updatesResponse.isOk(), "not ok response");
            return updatesResponse.getResult();
        } catch (Exception e) {
            throw new IllegalStateException("Can't parse JSON from: " + response, e);
        }
    }

    private static SendResponse parseSendMessage(String response) {
        try {
            return mapper.readValue(response, SendResponse.class);
        } catch (IOException e) {
            throw new RuntimeException("can't parse response", e);
        }
    }

    private final class Endpoint {
        final URI uri;
        final String endpoint;
        final AsyncMetrics callMetrics;
        final MetricRegistry reg;

        Endpoint(MetricRegistry registry, String uriPrefix, String endpoint) {
            this.uri = URI.create(uriPrefix + endpoint);
            this.endpoint = uri.getHost() + endpoint;
            this.reg = registry.subRegistry("endpoint", this.endpoint);
            this.callMetrics = new AsyncMetrics(this.reg, "http.client.call");
        }

        void incStatus(int code) {
            reg.rate("http.client.call.status", Labels.of("status", String.valueOf(code))).inc();
        }

        public CompletableFuture<HttpResponse<String>> send(HttpRequest.Builder builder) {
            return callMetrics.wrapFuture(() -> httpClient
                            .sendAsync(builder.uri(uri).build(), HttpResponse.BodyHandlers.ofString())
                            .orTimeout(DEFAULT_TIMEOUT_SECONDS, TimeUnit.SECONDS))
                .whenComplete((response, e) -> {
                    if (e != null) {
                        logger.info("{} failed", endpoint, e);
                        recreateHttpClient();
                        return;
                    }

                    this.incStatus(response.statusCode());
                    if (response.statusCode() != 200) {
                        logger.info("{} response {} != 200, body {}", endpoint, response.statusCode(), response.body());
                    }
                });
        }

        @Override
        public String toString() {
            return uri.toString();
        }
    }
}
