package ru.yandex.solomon.alert.telegram;

import java.net.http.HttpRequest;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.NoSuchElementException;
import java.util.UUID;


/**
 * @author alexlovkov
 **/
public class MultiPartBodyPublisher {

    private static final int MAX_LENGTH = 8192;

    private List<PartsSpecification> partsSpecificationList = new ArrayList<>();
    private String boundary = UUID.randomUUID().toString();

    public HttpRequest.BodyPublisher build() {
        if (partsSpecificationList.size() == 0) {
            throw new IllegalStateException("Must have at least one part to build multipart message.");
        }
        addFinalBoundaryPart();
        return HttpRequest.BodyPublishers.ofByteArrays(PartsIterator::new);
    }

    public String getBoundary() {
        return boundary;
    }

    public MultiPartBodyPublisher addPart(String name, String value) {
        PartsSpecification newPart = new PartsSpecification();
        newPart.type = PartsSpecification.TYPE.STRING;
        newPart.name = name;
        newPart.value = value;
        partsSpecificationList.add(newPart);
        return this;
    }

    public MultiPartBodyPublisher addPart(String name, byte[] bytes) {
        PartsSpecification newPart = new PartsSpecification();
        newPart.type = PartsSpecification.TYPE.BYTES;
        newPart.name = name;
        newPart.bytes = bytes;
        partsSpecificationList.add(newPart);
        return this;
    }

    private void addFinalBoundaryPart() {
        PartsSpecification newPart = new PartsSpecification();
        newPart.type = PartsSpecification.TYPE.FINAL_BOUNDARY;
        newPart.value = "--" + boundary + "--";
        partsSpecificationList.add(newPart);
    }

    static class PartsSpecification {

        public enum TYPE {
            STRING,
            FINAL_BOUNDARY,
            BYTES
        }

        PartsSpecification.TYPE type;
        String name;
        String value;
        byte[] bytes;
    }

    class PartsIterator implements Iterator<byte[]> {

        private Iterator<PartsSpecification> iter;
        private byte[] bytes;

        private boolean done;
        private byte[] next;

        PartsIterator() {
            iter = partsSpecificationList.iterator();
        }

        @Override
        public boolean hasNext() {
            if (done) {
                return false;
            }
            if (next != null) {
                return true;
            }
            next = computeNext();
            if (next == null) {
                done = true;
                return false;
            }
            return true;
        }

        @Override
        public byte[] next() {
            if (!hasNext()) {
                throw new NoSuchElementException();
            }
            byte[] res = next;
            next = null;
            return res;
        }

        private byte[] computeNext() {
            if (bytes == null) {
                if (!iter.hasNext()) {
                    return null;
                }
                PartsSpecification nextPart = iter.next();

                switch (nextPart.type) {
                    case STRING:
                        String part =
                            "--" + boundary + "\r\n" +
                                "Content-Disposition: form-data; name=" + nextPart.name + "\r\n" +
                                "Content-Type: text/plain; charset=UTF-8\r\n\r\n" +
                                nextPart.value + "\r\n";
                        return part.getBytes(StandardCharsets.UTF_8);
                    case FINAL_BOUNDARY:
                        return nextPart.value.getBytes(StandardCharsets.UTF_8);
                    case BYTES:
                        bytes = nextPart.bytes;
                        String partHeader =
                            "--" + boundary + "\r\n" +
                                "Content-Disposition: form-data; name=" + nextPart.name + "; filename=" + "image.png" +
                                "\r\n" +
                                "Content-Type: application/octet-stream\r\n\r\n";
                        return partHeader.getBytes(StandardCharsets.UTF_8);
                    default:
                        throw new IllegalStateException("can't handle type:" + nextPart.type);
                }
            } else {
                if (bytes.length < MAX_LENGTH) {
                    byte[] actualBytes = new byte[bytes.length + 2];
                    System.arraycopy(bytes, 0, actualBytes, 0, bytes.length);
                    byte[] end = "\r\n" .getBytes(StandardCharsets.UTF_8);
                    actualBytes[actualBytes.length - 2] = end[0];
                    actualBytes[actualBytes.length - 1] = end[1];
                    this.bytes = null;
                    return actualBytes;
                } else {
                    byte[] buf = new byte[MAX_LENGTH];
                    System.arraycopy(bytes, 0, buf, 0, MAX_LENGTH);
                    byte[] newBytes = new byte[bytes.length - MAX_LENGTH];
                    System.arraycopy(bytes, MAX_LENGTH, newBytes, 0, bytes.length - MAX_LENGTH);
                    bytes = newBytes;
                    return buf;
                }
            }
        }
    }
}
