package ru.yandex.solomon.alert.template;

import java.time.Clock;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.function.Predicate;
import java.util.stream.Collectors;

import io.grpc.Status;
import io.grpc.StatusRuntimeException;
import org.apache.commons.lang3.StringUtils;

import ru.yandex.solomon.alert.api.converters.AlertConverter;
import ru.yandex.solomon.alert.dao.AlertTemplateDao;
import ru.yandex.solomon.alert.dao.AlertTemplateLastVersionDao;
import ru.yandex.solomon.alert.protobuf.AlertTemplateStatus;
import ru.yandex.solomon.alert.protobuf.CreateAlertTemplateRequest;
import ru.yandex.solomon.alert.protobuf.CreateAlertTemplateResponse;
import ru.yandex.solomon.alert.protobuf.DeleteAlertTemplatePublicationRequest;
import ru.yandex.solomon.alert.protobuf.DeleteAlertTemplatePublicationResponse;
import ru.yandex.solomon.alert.protobuf.DeployAlertTemplateRequest;
import ru.yandex.solomon.alert.protobuf.DeployAlertTemplateResponse;
import ru.yandex.solomon.alert.protobuf.ListAlertTemplateRequest;
import ru.yandex.solomon.alert.protobuf.ListAlertTemplateResponse;
import ru.yandex.solomon.alert.protobuf.ListAlertTemplateVersionsRequest;
import ru.yandex.solomon.alert.protobuf.ListAlertTemplateVersionsResponse;
import ru.yandex.solomon.alert.protobuf.PublishAlertTemplateRequest;
import ru.yandex.solomon.alert.protobuf.PublishAlertTemplateResponse;
import ru.yandex.solomon.alert.protobuf.ReadAlertTemplateRequest;
import ru.yandex.solomon.alert.protobuf.ReadAlertTemplateResponse;
import ru.yandex.solomon.alert.protobuf.TemplateDeployPolicy;
import ru.yandex.solomon.alert.tasks.PublishAlertTemplateTaskHandler;
import ru.yandex.solomon.alert.template.domain.AlertTemplate;
import ru.yandex.solomon.alert.template.domain.AlertTemplateId;
import ru.yandex.solomon.alert.template.domain.AlertTemplateLastVersion;
import ru.yandex.solomon.core.exceptions.ConflictException;
import ru.yandex.solomon.scheduler.Task;
import ru.yandex.solomon.scheduler.TaskScheduler;
import ru.yandex.solomon.util.collection.Nullables;

/**
 * @author Alexey Trushkin
 */
public class AlertTemplateServiceImpl implements AlertTemplateService {
    private final AlertTemplateDao cachedAlertTemplateDao;
    private final AlertTemplateLastVersionDao alertTemplateLastVersionDao;
    private final TaskScheduler scheduler;
    private final Clock clock;

    public AlertTemplateServiceImpl(
            AlertTemplateDao alertTemplateDao,
            AlertTemplateLastVersionDao alertTemplateLastVersionDao,
            TaskScheduler scheduler,
            Clock clock)
    {
        this.cachedAlertTemplateDao = alertTemplateDao;
        this.alertTemplateLastVersionDao = alertTemplateLastVersionDao;
        this.scheduler = scheduler;
        this.clock = clock;
    }

    @Override
    public CompletableFuture<CreateAlertTemplateResponse> create(CreateAlertTemplateRequest request) {
        var now = clock.instant();
        var template = AlertConverter.alertTemplateFromProto(request.getAlertTemplate())
                .toBuilder()
                .setCreatedAt(now)
                .setUpdatedAt(now)
                .build();
        return cachedAlertTemplateDao.findById(template.getId(), template.getTemplateVersionTag())
                .thenCompose(alertTemplate -> {
                    if (alertTemplate.isPresent()) {
                        return CompletableFuture.failedFuture(
                                failure(Status.INVALID_ARGUMENT, "Alert template with id " + template.getId() + " version " + template.getTemplateVersionTag() + " already exists")
                        );
                    }
                    return cachedAlertTemplateDao.create(template)
                            .thenApply(aBoolean -> {
                                if (!aBoolean) {
                                    throw new ConflictException("AlertTemplate creation failed with conflict");
                                }
                                return CreateAlertTemplateResponse.newBuilder()
                                        .setAlertTemplate(AlertConverter.alertTemplateToProto(template, AlertTemplateStatus.ALERT_TEMPLATE_STATUS_DRAFT))
                                        .build();
                            });
                });
    }

    @Override
    public CompletableFuture<ReadAlertTemplateResponse> read(ReadAlertTemplateRequest request) {
        return alertTemplateLastVersionDao.findById(request.getTemplateId())
                .thenCompose(alertTemplateLastVersion -> {
                    if (StringUtils.isEmpty(request.getTemplateVersionTag()) && alertTemplateLastVersion.isEmpty()) {
                        throw failure(Status.NOT_FOUND, "Can't find alert template with id " + request.getTemplateId() + " version " + request.getTemplateVersionTag());
                    }
                    var publishedVersion = alertTemplateLastVersion.isPresent()
                            ? alertTemplateLastVersion.get().templateVersionTag()
                            : "";
                    var version = StringUtils.isEmpty(request.getTemplateVersionTag())
                            ? publishedVersion
                            : request.getTemplateVersionTag();
                    return find(request.getTemplateId(), version, publishedVersion);
                });
    }

    @Override
    public CompletableFuture<DeleteAlertTemplatePublicationResponse> unpublish(DeleteAlertTemplatePublicationRequest request) {
        return alertTemplateLastVersionDao.findById(request.getTemplateId())
                .thenCompose(version -> {
                    if (version.isEmpty()) {
                        throw failure(Status.NOT_FOUND, "Can't find published alert template with id " + request.getTemplateId());
                    }
                    AlertTemplateLastVersion alertTemplateLastVersion = version.get();
                    String taskId = Nullables.orEmpty(alertTemplateLastVersion.publishingTaskId());
                    if (taskId.isEmpty()) {
                        return alertTemplateLastVersionDao.delete(request.getTemplateId(), alertTemplateLastVersion.version());
                    }
                    return scheduler.getTask(taskId)
                            .thenCompose(taskOptional -> {
                                var state = taskOptional.map(Task::state).orElse(Task.State.COMPLETED);
                                if (state != Task.State.COMPLETED) {
                                    throw failure(Status.INVALID_ARGUMENT, "Another version of alert template with id " + request.getTemplateId() + " is deploying");
                                }
                                return alertTemplateLastVersionDao.delete(request.getTemplateId(), alertTemplateLastVersion.version());
                            });
                })
                .thenApply(aBoolean -> {
                    if (!aBoolean) {
                        throw new ConflictException("AlertTemplate deploy failed with conflict");
                    }
                    return DeleteAlertTemplatePublicationResponse.newBuilder().build();
                });
    }

    @Override
    public CompletableFuture<PublishAlertTemplateResponse> publish(PublishAlertTemplateRequest request) {
        return getAlertTemplate(request)
                .thenCompose(this::getAlertData)
                .thenCompose(this::validateTemplatePublishing)
                .thenCompose(this::publish);
    }

    private CompletableFuture<AlertData> validateTemplatePublishing(AlertData alertData) {
        String taskId = alertData.version.map(AlertTemplateLastVersion::publishingTaskId).orElse("");
        if (taskId.isEmpty()) {
            return CompletableFuture.completedFuture(alertData);
        }
        return scheduler.getTask(taskId)
                .thenApply(taskOptional -> {
                    var state = taskOptional.map(Task::state).orElse(Task.State.COMPLETED);
                    if (state != Task.State.COMPLETED) {
                        throw failure(Status.INVALID_ARGUMENT, "Another version of alert template with id " + alertData.alertTemplate.getId() + " is deploying");
                    }
                    return alertData;
                });
    }

    private CompletableFuture<PublishAlertTemplateResponse> publish(AlertData data) {
        int version = data.version.map(AlertTemplateLastVersion::version).orElse(-1);
        return cachedAlertTemplateDao.publish(data.alertTemplate, version)
                .thenApply(aBoolean -> {
                    if (!aBoolean) {
                        throw new ConflictException("AlertTemplate publication failed with conflict");
                    }
                    return PublishAlertTemplateResponse.newBuilder()
                            .setAlertTemplate(AlertConverter.alertTemplateToProto(data.alertTemplate, AlertTemplateStatus.ALERT_TEMPLATE_STATUS_PUBLISHED))
                            .build();
                });
    }

    private CompletableFuture<AlertData> getAlertData(AlertTemplate alertTemplate) {
        return alertTemplateLastVersionDao.findById(alertTemplate.getId())
                .thenApply(alertTemplateLastVersion -> new AlertData(alertTemplate, alertTemplateLastVersion));
    }

    private CompletableFuture<AlertTemplate> getAlertTemplate(PublishAlertTemplateRequest request) {
        return cachedAlertTemplateDao.findById(request.getTemplateId(), request.getTemplateVersionTag())
                .thenApply(alertTemplate -> {
                    if (alertTemplate.isEmpty()) {
                        throw failure(Status.NOT_FOUND, "Can't find alert template with id " + request.getTemplateId() + " version " + request.getTemplateVersionTag());
                    }
                    return alertTemplate.get();
                });
    }

    private CompletableFuture<ReadAlertTemplateResponse> find(String id, String templateVersionTag, String publishedVersion) {
        return cachedAlertTemplateDao.findById(id, templateVersionTag)
                .handle((alertTemplate, e) -> {
                    if (e != null) {
                        throw failure(Status.INTERNAL.withCause(e), "Failed to read template with id" + id + " version " + templateVersionTag);
                    }
                    if (alertTemplate.isEmpty()) {
                        throw failure(Status.NOT_FOUND, "Can't find alert template with id " + id + " version " + templateVersionTag);
                    }
                    var status = publishedVersion.equals(alertTemplate.get().getTemplateVersionTag())
                            ? AlertTemplateStatus.ALERT_TEMPLATE_STATUS_PUBLISHED
                            : AlertTemplateStatus.ALERT_TEMPLATE_STATUS_DRAFT;
                    return ReadAlertTemplateResponse.newBuilder()
                            .setAlertTemplate(AlertConverter.alertTemplateToProto(alertTemplate.get(), status))
                            .build();
                });
    }

    @Override
    public CompletableFuture<ListAlertTemplateResponse> list(ListAlertTemplateRequest request) {
        return alertTemplateLastVersionDao.getAll()
                .thenCompose(versions -> {
                    var publishIds = versions.stream()
                            .map(version -> new AlertTemplateId(version.id(), version.templateVersionTag()))
                            .collect(Collectors.toSet());
                    Predicate<AlertTemplate> skipIds = alertTemplate -> false;
                    if (request.getAlertTemplateStatusesFilterList().isEmpty()
                            || (request.getAlertTemplateStatusesFilterList().contains(AlertTemplateStatus.ALERT_TEMPLATE_STATUS_PUBLISHED)
                            && !request.getAlertTemplateStatusesFilterList().contains(AlertTemplateStatus.ALERT_TEMPLATE_STATUS_DRAFT))) {
                        skipIds = alertTemplate -> !publishIds.contains(alertTemplate.getCompositeId());
                    } else if (!request.getAlertTemplateStatusesFilterList().contains(AlertTemplateStatus.ALERT_TEMPLATE_STATUS_PUBLISHED) &&
                            request.getAlertTemplateStatusesFilterList().contains(AlertTemplateStatus.ALERT_TEMPLATE_STATUS_DRAFT)) {
                        skipIds = alertTemplate -> publishIds.contains(alertTemplate.getCompositeId());
                    }
                    return cachedAlertTemplateDao.listLastTemplates(
                            request.getServiceProviderId(),
                            request.getNameFilter(),
                            request.getLabelsSelector(),
                            (int) request.getPageSize(),
                            request.getPageToken(),
                            skipIds).handle((page, e) -> {
                                if (e != null) {
                                    throw failure(Status.INTERNAL.withCause(e), "Failed to read templates");
                                }
                                return ListAlertTemplateResponse.newBuilder()
                                        .addAllAlertTemplates(page.getItems().stream()
                                                .map(alertTemplate -> AlertConverter.alertTemplateToProto(alertTemplate,
                                                        publishIds.contains(alertTemplate.getCompositeId())
                                                                ? AlertTemplateStatus.ALERT_TEMPLATE_STATUS_PUBLISHED
                                                                : AlertTemplateStatus.ALERT_TEMPLATE_STATUS_DRAFT))
                                                .collect(Collectors.toList()))
                                        .setNextPageToken(page.getNextPageToken())
                                        .build();
                            });
                });
    }

    @Override
    public CompletableFuture<ListAlertTemplateVersionsResponse> listTemplateVersions(ListAlertTemplateVersionsRequest request) {
        return alertTemplateLastVersionDao.findById(request.getTemplateId())
                .thenCompose(version -> {
                    var publishedVersion = version.map(AlertTemplateLastVersion::templateVersionTag).orElse("");
                    return cachedAlertTemplateDao.listTemplateVersions(request.getTemplateId(), (int) request.getPageSize(), request.getPageToken())
                            .thenApply(page -> ListAlertTemplateVersionsResponse.newBuilder()
                                    .setNextPageToken(page.getNextPageToken())
                                    .addAllAlertTemplates(page.getItems().stream()
                                            .map(alertTemplate -> AlertConverter.alertTemplateToProto(alertTemplate, publishedVersion.equals(alertTemplate.getTemplateVersionTag())
                                                    ? AlertTemplateStatus.ALERT_TEMPLATE_STATUS_PUBLISHED
                                                    : AlertTemplateStatus.ALERT_TEMPLATE_STATUS_DRAFT))
                                            .collect(Collectors.toList()))
                                    .build());
                });
    }

    @Override
    public CompletableFuture<DeployAlertTemplateResponse> deploy(DeployAlertTemplateRequest request) {
        if (request.getTemplateDeployPolicy() == TemplateDeployPolicy.TEMPLATE_DEPLOY_POLICY_MANUAL) {
            return manualPolicy(request);
        }
        if (request.getTemplateDeployPolicy() == TemplateDeployPolicy.TEMPLATE_DEPLOY_POLICY_AUTO) {
            var taskId = PublishAlertTemplateTaskHandler.taskId();
            return autoPolicy(request, taskId);
        }
        return CompletableFuture.failedFuture(failure(Status.INTERNAL, "Deploy policy not implemented " + request.getTemplateDeployPolicy()));
    }

    private CompletableFuture<DeployAlertTemplateResponse> autoPolicy(DeployAlertTemplateRequest request, String taskId) {
        return alertTemplateLastVersionDao.findById(request.getTemplateId())
                .thenCompose(versionOptional -> {
                    if (versionOptional.isEmpty()) {
                        return CompletableFuture.failedFuture(failure(Status.NOT_FOUND, "Can't find alert template version data for id " + request.getTemplateId() + ", try to publish version before"));
                    }
                    int version = versionOptional.get().version();
                    return scheduler.getTask(versionOptional.get().publishingTaskId())
                            .thenCompose(taskOptional -> {
                                var state = taskOptional.map(Task::state).orElse(Task.State.COMPLETED);
                                if (state != Task.State.COMPLETED) {
                                    return CompletableFuture.failedFuture(failure(Status.INVALID_ARGUMENT, "Another version of alert template with id " + request.getTemplateId() + " is deploying"));
                                }
                                return alertTemplateLastVersionDao.updateDeployTask(request.getTemplateId(), version, taskId)
                                        .thenApply(aBoolean -> {
                                            if (!aBoolean) {
                                                throw new ConflictException("AlertTemplate deploy failed with conflict");
                                            }
                                            return versionOptional.get();
                                        });
                            });
                })
                .thenCompose(version -> scheduler.schedule(PublishAlertTemplateTaskHandler.task(request.getTemplateId(), request.getTemplateVersionTag(), taskId))
                        .thenApply(unused2 -> DeployAlertTemplateResponse.newBuilder()
                                .setTaskId(taskId)
                                .build()));
    }

    private CompletableFuture<DeployAlertTemplateResponse> manualPolicy(DeployAlertTemplateRequest request) {
        return alertTemplateLastVersionDao.findById(request.getTemplateId())
                .thenCompose(versionOptional -> {
                    if (versionOptional.isEmpty()) {
                        return CompletableFuture.failedFuture(failure(Status.NOT_FOUND, "Can't find alert template version data for id " + request.getTemplateId() + ", try to publish version before"));
                    }
                    return CompletableFuture.completedFuture(DeployAlertTemplateResponse.newBuilder().build());
                });
    }

    private StatusRuntimeException failure(Status status, String msg) {
        return status.withDescription(msg).asRuntimeException();
    }

    private record AlertData(AlertTemplate alertTemplate, Optional<AlertTemplateLastVersion> version) {
    }
}
