package ru.yandex.solomon.alert.template.domain;

import java.time.Instant;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.StringJoiner;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.google.common.collect.ImmutableMap;
import org.apache.commons.lang3.builder.EqualsBuilder;
import org.apache.commons.lang3.builder.EqualsExclude;

import ru.yandex.solomon.alert.domain.AlertSeverity;
import ru.yandex.solomon.alert.domain.NoPointsPolicy;
import ru.yandex.solomon.alert.domain.ResolvedEmptyPolicy;
import ru.yandex.solomon.util.collection.Nullables;

/**
 * @author Alexey Trushkin
 */
public abstract class AbstractAlertTemplate implements AlertTemplate {

    // PK fields
    private final String id;
    private final String templateVersionTag;

    private final String serviceProviderId;
    private final List<AlertTemplateParameter> parameters;
    private final List<AlertTemplateParameter> thresholds;
    private final String name;
    private final String description;
    private final List<String> groupByLabels;
    private final Map<String, String> annotations;
    private final Map<String, String> labels;
    private final int periodMillis;
    private final int delaySeconds;
    private final boolean defaultTemplate;
    private final ResolvedEmptyPolicy resolvedEmptyPolicy;
    private final NoPointsPolicy noPointsPolicy;
    private final AlertSeverity alertSeverity;

    @EqualsExclude
    private final String createdBy;
    @EqualsExclude
    private final String updatedBy;
    @EqualsExclude
    private final Instant createdAt;
    @EqualsExclude
    private final Instant updatedAt;

    protected AbstractAlertTemplate(AbstractAlertTemplateBuilder<?, ?> builder) {
        id = Objects.requireNonNull(builder.id, "id");
        templateVersionTag = Objects.requireNonNull(builder.templateVersionTag, "templateVersionTag");
        serviceProviderId = Objects.requireNonNull(builder.serviceProviderId, "serviceProviderId");
        description = Nullables.orEmpty(builder.description);
        name = Nullables.orEmpty(builder.name);
        parameters = Nullables.orEmpty(builder.parameters);
        thresholds = Nullables.orEmpty(builder.thresholds);
        groupByLabels = Nullables.orEmpty(builder.groupByLabels);
        defaultTemplate = Nullables.orFalse(builder.defaultTemplate);
        this.annotations = ImmutableMap.copyOf(builder.annotations);
        this.labels = ImmutableMap.copyOf(builder.labels);
        this.periodMillis = builder.periodMillis;
        this.delaySeconds = builder.delaySeconds;
        this.resolvedEmptyPolicy = builder.resolvedEmptyPolicy;
        this.noPointsPolicy = builder.noPointsPolicy;
        this.alertSeverity = builder.alertSeverity;
        this.createdBy = Nullables.orEmpty(builder.createdBy);
        this.createdAt = builder.createdAt != null ? builder.createdAt : Instant.now();
        this.updatedBy = Nullables.orEmpty(builder.updatedBy);
        this.updatedAt = builder.updatedAt != null ? builder.updatedAt : Instant.now();
    }

    public String getId() {
        return id;
    }

    @Override
    public String getTemplateVersionTag() {
        return templateVersionTag;
    }

    @Override
    public String getServiceProviderId() {
        return serviceProviderId;
    }

    @Override
    public String getName() {
        return name;
    }

    @Override
    public String getDescription() {
        return description;
    }

    @Override
    public List<String> getGroupByLabels() {
        return groupByLabels;
    }

    @Override
    public abstract AlertTemplateType getAlertTemplateType();

    @Override
    public Map<String, String> getAnnotations() {
        return annotations;
    }

    @Override
    public Map<String, String> getLabels() {
        return labels;
    }

    @Override
    public int getPeriodMillis() {
        return periodMillis;
    }

    @Override
    public boolean isDefaultTemplate() {
        return defaultTemplate;
    }

    @Override
    public int getDelaySeconds() {
        return delaySeconds;
    }

    @Override
    public ResolvedEmptyPolicy getResolvedEmptyPolicy() {
        return resolvedEmptyPolicy;
    }

    @Override
    public NoPointsPolicy getNoPointsPolicy() {
        return noPointsPolicy;
    }

    @Override
    public AlertSeverity getSeverity() {
        return alertSeverity;
    }

    public String getCreatedBy() {
        return createdBy;
    }

    public String getUpdatedBy() {
        return updatedBy;
    }

    public Instant getCreatedAt() {
        return createdAt;
    }

    public Instant getUpdatedAt() {
        return updatedAt;
    }

    @Override
    public final boolean equalContent(AlertTemplate alert) {
        return EqualsBuilder.reflectionEquals(this, alert);
    }

    @Override
    public abstract AbstractAlertTemplateBuilder<?, ?> toBuilder();

    @Override
    public List<AlertTemplateParameter> getThresholds() {
        return thresholds;
    }

    @Override
    public List<AlertTemplateParameter> getParameters() {
        return parameters;
    }

    @Override
    @JsonIgnore
    public AlertTemplateId getCompositeId() {
        return new AlertTemplateId(id, templateVersionTag);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        AbstractAlertTemplate that = (AbstractAlertTemplate) o;
        return periodMillis == that.periodMillis &&
                delaySeconds == that.delaySeconds &&
                Objects.equals(id, that.id) &&
                Objects.equals(templateVersionTag, that.templateVersionTag) &&
                Objects.equals(serviceProviderId, that.serviceProviderId) &&
                Objects.equals(parameters, that.parameters) &&
                Objects.equals(thresholds, that.thresholds) &&
                Objects.equals(name, that.name) &&
                Objects.equals(description, that.description) &&
                Objects.equals(groupByLabels, that.groupByLabels) &&
                Objects.equals(annotations, that.annotations) &&
                Objects.equals(labels, that.labels) &&
                resolvedEmptyPolicy == that.resolvedEmptyPolicy &&
                noPointsPolicy == that.noPointsPolicy &&
                alertSeverity == that.alertSeverity &&
                Objects.equals(defaultTemplate, that.defaultTemplate) &&
                Objects.equals(createdBy, that.createdBy) &&
                Objects.equals(updatedBy, that.updatedBy) &&
                Objects.equals(createdAt, that.createdAt) &&
                Objects.equals(updatedAt, that.updatedAt);
    }

    @Override
    public int hashCode() {
        return Objects.hash(id, templateVersionTag, serviceProviderId, parameters,
                thresholds, name, description, groupByLabels, annotations, labels, periodMillis, delaySeconds,
                resolvedEmptyPolicy, noPointsPolicy, createdBy, updatedBy, createdAt, updatedAt, defaultTemplate,
                alertSeverity);
    }

    @Override
    public String toString() {
        return new StringJoiner(", ", AbstractAlertTemplate.class.getSimpleName() + "[", "]")
                .add("id='" + id + "'")
                .add("templateVersion='" + templateVersionTag + "'")
                .add("serviceProviderId='" + serviceProviderId + "'")
                .add("parameters=" + parameters)
                .add("thresholds=" + thresholds)
                .add("name='" + name + "'")
                .add("description='" + description + "'")
                .add("groupByLabels=" + groupByLabels)
                .add("annotations=" + annotations)
                .add("labels=" + labels)
                .add("periodMillis=" + periodMillis)
                .add("delaySeconds=" + delaySeconds)
                .add("resolvedEmptyPolicy=" + resolvedEmptyPolicy)
                .add("noPointsPolicy=" + noPointsPolicy)
                .add("alertSeverity=" + alertSeverity)
                .add("createdBy='" + createdBy + "'")
                .add("updatedBy='" + updatedBy + "'")
                .add("createdAt=" + createdAt)
                .add("updatedAt=" + updatedAt)
                .add("defaultTemplate=" + defaultTemplate)
                .toString();
    }
}
