package ru.yandex.solomon.alert.template.domain;

import java.time.Instant;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import java.util.stream.StreamSupport;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableMap;

import ru.yandex.solomon.alert.domain.AlertSeverity;
import ru.yandex.solomon.alert.domain.NoPointsPolicy;
import ru.yandex.solomon.alert.domain.ResolvedEmptyPolicy;
import ru.yandex.solomon.alert.domain.StringInterner;

/**
 * @author Alexey Trushkin
 */
@ParametersAreNonnullByDefault
public abstract class AbstractAlertTemplateBuilder<AlertTemplateType extends AlertTemplate, BuilderType extends AbstractAlertTemplateBuilder<AlertTemplateType, BuilderType>> {
    @Nullable
    protected String id;
    @Nullable
    protected String templateVersionTag;
    @Nullable
    String serviceProviderId;
    @Nullable
    String name;
    @Nullable
    String description;
    @Nullable
    String createdBy;
    @Nullable
    Instant createdAt;
    @Nullable
    String updatedBy;
    @Nullable
    Instant updatedAt;
    List<String> groupByLabels = Collections.emptyList();
    List<AlertTemplateParameter> parameters = Collections.emptyList();
    List<AlertTemplateParameter> thresholds = Collections.emptyList();
    Map<String, String> annotations = ImmutableMap.of();
    Map<String, String> labels = ImmutableMap.of();
    int periodMillis = 0;
    int delaySeconds = 0;
    boolean defaultTemplate = false;
    ResolvedEmptyPolicy resolvedEmptyPolicy = ResolvedEmptyPolicy.DEFAULT;
    NoPointsPolicy noPointsPolicy = NoPointsPolicy.DEFAULT;
    AlertSeverity alertSeverity = AlertSeverity.UNKNOWN;

    public AbstractAlertTemplateBuilder() {
    }

    public AbstractAlertTemplateBuilder(AlertTemplateType template) {
        this.id = template.getId();
        this.templateVersionTag = template.getTemplateVersionTag();
        this.serviceProviderId = template.getServiceProviderId();
        this.name = template.getName();
        this.description = template.getDescription();
        this.createdBy = template.getCreatedBy();
        this.createdAt = template.getCreatedAt();
        this.updatedBy = template.getUpdatedBy();
        this.updatedAt = template.getUpdatedAt();
        this.groupByLabels = template.getGroupByLabels();
        this.annotations = template.getAnnotations();
        this.labels = template.getLabels();
        this.periodMillis = template.getPeriodMillis();
        this.delaySeconds = template.getDelaySeconds();
        this.defaultTemplate = template.isDefaultTemplate();
        this.resolvedEmptyPolicy = template.getResolvedEmptyPolicy();
        this.noPointsPolicy = template.getNoPointsPolicy();
        this.alertSeverity = template.getSeverity();
        this.parameters = template.getParameters();
        this.thresholds = template.getThresholds();
    }

    protected abstract BuilderType self();

    public BuilderType setId(String id) {
        this.id = id;
        return self();
    }

    public BuilderType setTemplateVersionTag(String templateVersionTag) {
        this.templateVersionTag = StringInterner.I.intern(templateVersionTag);
        return self();
    }

    public BuilderType setServiceProviderId(String serviceProviderId) {
        this.serviceProviderId = StringInterner.I.intern(serviceProviderId);
        return self();
    }

    public BuilderType setName(String name) {
        this.name = name;
        return self();
    }

    public BuilderType setDescription(String description) {
        this.description = description;
        return self();
    }

    public BuilderType setCreatedBy(String user) {
        this.createdBy = StringInterner.I.intern(user);
        return self();
    }

    public BuilderType setCreatedAt(Instant createdAt) {
        this.createdAt = createdAt;
        return self();
    }

    public BuilderType setUpdatedBy(String user) {
        this.updatedBy = StringInterner.I.intern(user);
        return self();
    }

    public BuilderType setUpdatedAt(Instant updatedAt) {
        this.updatedAt = updatedAt;
        return self();
    }

    public BuilderType setGroupByLabels(Iterable<String> labels) {
        this.groupByLabels = StreamSupport.stream(labels.spliterator(), false)
                .map(StringInterner.I::intern)
                .collect(Collectors.toUnmodifiableList());
        return self();
    }

    public BuilderType setGroupByLabel(String label) {
        this.groupByLabels = List.of(StringInterner.I.intern(label));
        return self();
    }

    public BuilderType setAnnotations(Map<String, String> annotations) {
        var map = ImmutableMap.<String, String>builder();
        for (var entry : annotations.entrySet()) {
            map.put(StringInterner.I.intern(entry.getKey()), StringInterner.I.intern(entry.getValue()));
        }
        this.annotations = map.build();
        return self();
    }

    public BuilderType addAnnotation(String key, String value) {
        this.annotations = ImmutableMap.<String, String>builder()
                .putAll(annotations)
                .put(StringInterner.I.intern(key), StringInterner.I.intern(value))
                .build();
        return self();
    }

    public BuilderType setLabels(Map<String, String> labels) {
        var map = ImmutableMap.<String, String>builder();
        for (var entry : labels.entrySet()) {
            map.put(StringInterner.I.intern(entry.getKey()), StringInterner.I.intern(entry.getValue()));
        }
        this.labels = map.build();
        return self();
    }

    public BuilderType addLabel(String key, String value) {
        this.labels = ImmutableMap.<String, String>builder()
                .putAll(labels)
                .put(StringInterner.I.intern(key), StringInterner.I.intern(value))
                .build();
        return self();
    }

    public BuilderType setPeriodMillis(int periodMillis) {
        this.periodMillis = periodMillis;
        return self();
    }

    public BuilderType setDelaySeconds(int delaySeconds) {
        this.delaySeconds = delaySeconds;
        return self();
    }

    public BuilderType setResolvedEmptyPolicy(ResolvedEmptyPolicy resolvedEmptyPolicy) {
        this.resolvedEmptyPolicy = resolvedEmptyPolicy;
        return self();
    }

    public BuilderType setNoPointsPolicy(NoPointsPolicy noPointsPolicy) {
        this.noPointsPolicy = noPointsPolicy;
        return self();
    }

    public BuilderType setAlertSeverity(AlertSeverity alertSeverity) {
        this.alertSeverity = alertSeverity;
        return self();
    }

    public BuilderType setParameters(Iterable<AlertTemplateParameter> parameters) {
        this.parameters = StreamSupport.stream(parameters.spliterator(), false)
                .collect(Collectors.toUnmodifiableList());
        return self();
    }

    public BuilderType setParameter(AlertTemplateParameter parameter) {
        this.parameters = List.of(parameter);
        return self();
    }

    public BuilderType setThresholds(Iterable<AlertTemplateParameter> parameters) {
        this.thresholds = StreamSupport.stream(parameters.spliterator(), false)
                .collect(Collectors.toUnmodifiableList());
        return self();
    }

    public BuilderType setThresholds(AlertTemplateParameter parameter) {
        this.thresholds = List.of(parameter);
        return self();
    }

    public BuilderType setDefaultTemplate(boolean defaultTemplate) {
        this.defaultTemplate = defaultTemplate;
        return self();
    }

    public abstract AlertTemplateType build();
}
