package ru.yandex.solomon.alert.template.domain;

import java.util.List;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.annotation.JsonSubTypes;
import com.fasterxml.jackson.annotation.JsonTypeInfo;
import org.apache.commons.lang3.StringUtils;

import ru.yandex.misc.lang.DefaultObject;
import ru.yandex.monitoring.api.v3.UnitFormat;
import ru.yandex.solomon.alert.protobuf.DoubleTemplateParameter;
import ru.yandex.solomon.alert.protobuf.IntegerTemplateParameter;
import ru.yandex.solomon.alert.protobuf.LabelListTemplateParameter;
import ru.yandex.solomon.alert.protobuf.TextListTemplateParameter;
import ru.yandex.solomon.alert.protobuf.TextTemplateParameter;
import ru.yandex.solomon.util.collection.Nullables;

/**
 * @author Alexey Trushkin
 */
@JsonTypeInfo(
        use = JsonTypeInfo.Id.NAME,
        property = "type")
@JsonSubTypes({
        @JsonSubTypes.Type(value = AlertTemplateParameter.TextParameterValue.class, name = "TEXT"),
        @JsonSubTypes.Type(value = AlertTemplateParameter.TextListParameterValue.class, name = "TEXT_LIST"),
        @JsonSubTypes.Type(value = AlertTemplateParameter.LabelListParameterValue.class, name = "LABEL_LIST"),
        @JsonSubTypes.Type(value = AlertTemplateParameter.DoubleParameterValue.class, name = "DOUBLE"),
        @JsonSubTypes.Type(value = AlertTemplateParameter.IntegerParameterValue.class, name = "INTEGER"),
})
public abstract class AlertTemplateParameter extends DefaultObject {

    @JsonProperty
    private final String name;
    @JsonProperty
    private final String title;
    @JsonProperty
    private final String description;

    @JsonCreator
    public AlertTemplateParameter(
            @JsonProperty("name") String name,
            @JsonProperty("title") String title,
            @JsonProperty("description") String description)
    {
        this.name = Nullables.orEmpty(name);
        this.title = Nullables.orEmpty(title);
        this.description = Nullables.orEmpty(description);
    }

    public String getName() {
        return name;
    }

    public String getTitle() {
        return title;
    }

    public String getDescription() {
        return description;
    }

    public abstract ParameterValueType getType();

    public abstract ru.yandex.solomon.alert.protobuf.AlertTemplateParameter toProto();

    public static AlertTemplateParameter fromProto(ru.yandex.solomon.alert.protobuf.AlertTemplateParameter param) {
        switch (param.getParameterCase()) {
            case DOUBLE_PARAMETER -> {
                return DoubleParameterValue.fromProto(param.getDoubleParameter());
            }
            case INTEGER_PARAMETER -> {
                return IntegerParameterValue.fromProto(param.getIntegerParameter());
            }
            case TEXT_PARAMETER -> {
                return TextParameterValue.fromProto(param.getTextParameter());
            }
            case TEXT_LIST_PARAMETER -> {
                return TextListParameterValue.fromProto(param.getTextListParameter());
            }
            case LABEL_LIST_PARAMETER -> {
                return LabelListParameterValue.fromProto(param.getLabelListParameter());
            }
            default -> throw new IllegalArgumentException("Unknown alert template parameter type " + param.getParameterCase());
        }
    }

    private static UnitFormat fromInternalProto(String unitFormat) {
        if (StringUtils.isEmpty(unitFormat)) {
            return UnitFormat.UNIT_FORMAT_UNSPECIFIED;
        }
        try {
            return UnitFormat.valueOf(unitFormat);
        } catch (Exception e) {
            return UnitFormat.UNIT_FORMAT_UNSPECIFIED;
        }
    }

    @JsonIgnore
    public abstract boolean isDefaultValueValid();

    public enum ParameterValueType {
        TEXT,
        TEXT_LIST,
        DOUBLE,
        INTEGER,
        LABEL_LIST,
    }

    public static class DoubleParameterValue extends AlertTemplateParameter {
        @JsonProperty
        private final double defaultValue;
        @JsonProperty
        private final String unit;

        @JsonCreator
        public DoubleParameterValue(
                @JsonProperty("defaultValue") Double defaultValue,
                @JsonProperty("name") String name,
                @JsonProperty("title") String title,
                @JsonProperty("description") String description,
                @JsonProperty("unit") String unit)
        {
            super(name, title, description);
            this.defaultValue = Nullables.orZero(defaultValue);
            this.unit = Nullables.orEmpty(unit);
        }

        public double getDefaultValue() {
            return defaultValue;
        }

        @Override
        public ParameterValueType getType() {
            return ParameterValueType.DOUBLE;
        }

        @Override
        public ru.yandex.solomon.alert.protobuf.AlertTemplateParameter toProto() {
            return ru.yandex.solomon.alert.protobuf.AlertTemplateParameter.newBuilder()
                    .setDoubleParameter(toProtoDouble())
                    .build();
        }

        public DoubleTemplateParameter toProtoDouble() {
            return DoubleTemplateParameter.newBuilder()
                    .setName(getName())
                    .setTitle(getTitle())
                    .setDefaultValue(getDefaultValue())
                    .setDescription(getDescription())
                    .setUnitFormat(AlertTemplateParameter.fromInternalProto(unit))
                    .build();
        }

        public static AlertTemplateParameter fromProto(DoubleTemplateParameter param) {
            return new DoubleParameterValue(
                    param.getDefaultValue(),
                    param.getName(),
                    param.getTitle(),
                    param.getDescription(),
                    param.getUnitFormat().toString());
        }

        @Override
        public boolean isDefaultValueValid() {
            return true;
        }
    }

    public static class IntegerParameterValue extends AlertTemplateParameter {
        @JsonProperty
        private final int defaultValue;
        @JsonProperty
        private final String unit;

        @JsonCreator
        public IntegerParameterValue(
                @JsonProperty("defaultValue") Integer defaultValue,
                @JsonProperty("name") String name,
                @JsonProperty("title") String title,
                @JsonProperty("description") String description,
                @JsonProperty("unit") String unit)
        {
            super(name, title, description);
            this.defaultValue = Nullables.orZero(defaultValue);
            this.unit = Nullables.orEmpty(unit);
        }

        public int getDefaultValue() {
            return defaultValue;
        }

        @Override
        public ru.yandex.solomon.alert.protobuf.AlertTemplateParameter toProto() {
            return ru.yandex.solomon.alert.protobuf.AlertTemplateParameter.newBuilder()
                    .setIntegerParameter(IntegerTemplateParameter.newBuilder()
                            .setName(getName())
                            .setTitle(getTitle())
                            .setDefaultValue(getDefaultValue())
                            .setDescription(getDescription())
                            .setUnitFormat(AlertTemplateParameter.fromInternalProto(unit))
                            .build())
                    .build();
        }

        public static AlertTemplateParameter fromProto(IntegerTemplateParameter param) {
            return new IntegerParameterValue(
                    (int) param.getDefaultValue(),
                    param.getName(),
                    param.getTitle(),
                    param.getDescription(),
                    param.getUnitFormat().toString());
        }

        @Override
        public ParameterValueType getType() {
            return ParameterValueType.INTEGER;
        }

        @Override
        public boolean isDefaultValueValid() {
            return true;
        }
    }

    public static class TextParameterValue extends AlertTemplateParameter {
        @JsonProperty
        private final String defaultValue;

        @JsonCreator
        public TextParameterValue(
                @JsonProperty("defaultValue") String defaultValue,
                @JsonProperty("name") String name,
                @JsonProperty("title") String title,
                @JsonProperty("description") String description)
        {
            super(name, title, description);
            this.defaultValue = Nullables.orEmpty(defaultValue);
        }

        public String getDefaultValue() {
            return defaultValue;
        }

        @Override
        public ru.yandex.solomon.alert.protobuf.AlertTemplateParameter toProto() {
            return ru.yandex.solomon.alert.protobuf.AlertTemplateParameter.newBuilder()
                    .setTextParameter(TextTemplateParameter.newBuilder()
                            .setName(getName())
                            .setTitle(getTitle())
                            .setDefaultValue(getDefaultValue())
                            .setDescription(getDescription())
                            .build())
                    .build();
        }

        public static AlertTemplateParameter fromProto(TextTemplateParameter param) {
            return new TextParameterValue(param.getDefaultValue(), param.getName(), param.getTitle(), param.getDescription());
        }

        @Override
        public ParameterValueType getType() {
            return ParameterValueType.TEXT;
        }

        @Override
        public boolean isDefaultValueValid() {
            return !defaultValue.isEmpty();
        }
    }

    public static class TextListParameterValue extends AlertTemplateParameter {
        @JsonProperty
        private final List<String> defaultValues;

        @JsonCreator
        public TextListParameterValue(
                @JsonProperty("defaultValues") List<String> defaultValues,
                @JsonProperty("name") String name,
                @JsonProperty("title") String title,
                @JsonProperty("description") String description)
        {
            super(name, title, description);
            this.defaultValues = Nullables.orEmpty(defaultValues);
        }

        public List<String> getDefaultValues() {
            return defaultValues;
        }

        @Override
        public ru.yandex.solomon.alert.protobuf.AlertTemplateParameter toProto() {
            return ru.yandex.solomon.alert.protobuf.AlertTemplateParameter.newBuilder()
                    .setTextListParameter(TextListTemplateParameter.newBuilder()
                            .setName(getName())
                            .setTitle(getTitle())
                            .addAllDefaultValues(getDefaultValues())
                            .setDescription(getDescription())
                            .build())
                    .build();
        }

        public static AlertTemplateParameter fromProto(TextListTemplateParameter param) {
            return new TextListParameterValue(param.getDefaultValuesList(), param.getName(), param.getTitle(), param.getDescription());
        }

        @Override
        public ParameterValueType getType() {
            return ParameterValueType.TEXT_LIST;
        }

        @Override
        public boolean isDefaultValueValid() {
            return !defaultValues.isEmpty();
        }
    }

    public static class LabelListParameterValue extends AlertTemplateParameter {
        @JsonProperty
        private final String selectors;
        @JsonProperty
        private final String labelKey;
        @JsonProperty
        private final List<String> defaultValues;
        @JsonProperty
        private final String projectId;
        @JsonProperty
        private final boolean multiselectable;

        @JsonCreator
        public LabelListParameterValue(
                @JsonProperty("selectors") String selectors,
                @JsonProperty("labelKey") String labelKey,
                @JsonProperty("defaultValues") List<String> defaultValues,
                @JsonProperty("name") String name,
                @JsonProperty("title") String title,
                @JsonProperty("description") String description,
                @JsonProperty("projectId") String projectId,
                @JsonProperty("multiselectable") Boolean multiselectable)
        {
            super(name, title, description);
            this.selectors = Nullables.orEmpty(selectors);
            this.labelKey = Nullables.orEmpty(labelKey);
            this.defaultValues = Nullables.orEmpty(defaultValues);
            this.projectId = Nullables.orEmpty(projectId);
            this.multiselectable = Nullables.orFalse(multiselectable);
        }

        public String getLabelKey() {
            return labelKey;
        }

        public String getSelectors() {
            return selectors;
        }

        public List<String> getDefaultValues() {
            return defaultValues;
        }

        public String getProjectId() {
            return projectId;
        }

        public boolean isMultiselectable() {
            return multiselectable;
        }

        @Override
        public ru.yandex.solomon.alert.protobuf.AlertTemplateParameter toProto() {
            return ru.yandex.solomon.alert.protobuf.AlertTemplateParameter.newBuilder()
                    .setLabelListParameter(LabelListTemplateParameter.newBuilder()
                            .setName(getName())
                            .setTitle(getTitle())
                            .setSelectors(getSelectors())
                            .setLabelKey(getLabelKey())
                            .addAllDefaultValues(getDefaultValues())
                            .setDescription(getDescription())
                            .setProjectId(getProjectId())
                            .setMultiselectable(isMultiselectable())
                            .build())
                    .build();
        }

        public static AlertTemplateParameter fromProto(LabelListTemplateParameter param) {
            return new LabelListParameterValue(
                    param.getSelectors(),
                    param.getLabelKey(),
                    param.getDefaultValuesList(),
                    param.getName(),
                    param.getTitle(),
                    param.getDescription(),
                    param.getProjectId(),
                    param.getMultiselectable());
        }

        @Override
        public ParameterValueType getType() {
            return ParameterValueType.LABEL_LIST;
        }

        @Override
        public boolean isDefaultValueValid() {
            return !defaultValues.isEmpty();
        }
    }

}
