package ru.yandex.solomon.alert.template.domain.threshold;

import java.util.Objects;

import javax.annotation.Nullable;

import com.fasterxml.jackson.annotation.JsonProperty;

import ru.yandex.solomon.alert.domain.threshold.Compare;
import ru.yandex.solomon.alert.domain.threshold.TargetStatus;
import ru.yandex.solomon.alert.domain.threshold.ThresholdType;
import ru.yandex.solomon.alert.protobuf.ECompare;
import ru.yandex.solomon.alert.protobuf.EThresholdType;
import ru.yandex.solomon.alert.protobuf.PredicateRule;
import ru.yandex.solomon.alert.protobuf.TPredicateRule;
import ru.yandex.solomon.util.collection.Nullables;

/**
 * @author Alexey Trushkin
 */
public class TemplatePredicateRule {
    private final ThresholdType thresholdType;
    private final double threshold;
    @JsonProperty
    private final String thresholdParameterTemplate;
    private final Compare comparison;
    private final TargetStatus status;

    public ThresholdType getThresholdType() {
        return thresholdType;
    }

    public double getThreshold() {
        return threshold;
    }

    public Compare getComparison() {
        return comparison;
    }

    public TargetStatus getTargetStatus() {
        return status;
    }

    private TemplatePredicateRule(@Nullable ThresholdType thresholdType,
                                  double threshold,
                                  @Nullable Compare comparison,
                                  @Nullable TargetStatus status,
                                  @Nullable String thresholdParameterTemplate)
    {
        this.thresholdType = Nullables.orDefault(thresholdType, ThresholdType.AT_LEAST_ONE);
        this.threshold = threshold;
        this.comparison = Nullables.orDefault(comparison, Compare.GTE);
        this.status = Nullables.orDefault(status, TargetStatus.ALARM);
        this.thresholdParameterTemplate = Nullables.orEmpty(thresholdParameterTemplate);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        TemplatePredicateRule that = (TemplatePredicateRule) o;
        return Double.compare(that.threshold, threshold) == 0 &&
                thresholdType == that.thresholdType &&
                comparison == that.comparison &&
                status == that.status &&
                Objects.equals(thresholdParameterTemplate, that.thresholdParameterTemplate);
    }

    public String getThresholdParameterTemplate() {
        return thresholdParameterTemplate;
    }

    public static TemplatePredicateRule of(ThresholdType thresholdType, Compare comparison, double threshold, TargetStatus status) {
        return new TemplatePredicateRule(thresholdType, threshold, comparison, status, null);
    }

    public static TemplatePredicateRule onThreshold(double threshold) {
        return new TemplatePredicateRule(null, threshold, null, null, null);
    }

    public TemplatePredicateRule withThreshold(double threshold) {
        return new TemplatePredicateRule(thresholdType, threshold, comparison, status, thresholdParameterTemplate);
    }

    public TemplatePredicateRule withThresholdType(ThresholdType thresholdType) {
        return new TemplatePredicateRule(thresholdType, threshold, comparison, status, thresholdParameterTemplate);
    }

    public TemplatePredicateRule withComparison(Compare comparison) {
        return new TemplatePredicateRule(thresholdType, threshold, comparison, status, thresholdParameterTemplate);
    }

    public TemplatePredicateRule withTargetStatus(TargetStatus status) {
        return new TemplatePredicateRule(thresholdType, threshold, comparison, status, thresholdParameterTemplate);
    }

    public TemplatePredicateRule onThresholdParameter(String thresholdParameterTemplate) {
        return new TemplatePredicateRule(thresholdType, threshold, comparison, status, thresholdParameterTemplate);
    }

    @Override
    public int hashCode() {
        return Objects.hash(thresholdType, threshold, comparison, status, thresholdParameterTemplate);
    }

    @Override
    public String toString() {
        return "TemplatePredicateRule{" +
                "thresholdType=" + thresholdType +
                ", threshold=" + threshold +
                ", comparison=" + comparison +
                ", status=" + status +
                ", thresholdParameterTemplate=" + thresholdParameterTemplate +
                '}';
    }

    public static PredicateRule toProto(TemplatePredicateRule predicateRule) {
        var builder = PredicateRule.newBuilder()
                .setComparison(ECompare.forNumber(predicateRule.getComparison().getNumber()))
                .setThresholdType(EThresholdType.forNumber(predicateRule.getThresholdType().getNumber()))
                .setThreshold(predicateRule.getThreshold())
                .setTargetStatus(TPredicateRule.ETargetStatus.forNumber(predicateRule.getTargetStatus().getNumber()))
                .setThresholdParameterTemplate(predicateRule.getThresholdParameterTemplate());
        return builder.build();
    }

    public static TemplatePredicateRule fromProto(PredicateRule predicateRule) {
        return new TemplatePredicateRule(
                ThresholdType.forNumber(predicateRule.getThresholdTypeValue()),
                predicateRule.getThreshold(),
                Compare.forNumber(predicateRule.getComparisonValue()),
                TargetStatus.forNumber(predicateRule.getTargetStatusValue()),
                predicateRule.getThresholdParameterTemplate()
        );
    }
}
