package ru.yandex.solomon.alert.template.domain.threshold;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.alert.template.domain.AbstractAlertTemplate;
import ru.yandex.solomon.alert.template.domain.AbstractAlertTemplateBuilder;
import ru.yandex.solomon.alert.template.domain.AlertTemplateType;

import static java.util.Objects.requireNonNull;

/**
 * @author Alexey Trushkin
 */
@ParametersAreNonnullByDefault
public class ThresholdAlertTemplate extends AbstractAlertTemplate {
    private final String selectors;
    private final String transformations;
    private final List<TemplatePredicateRule> predicateRules;

    protected ThresholdAlertTemplate(ThresholdAlertTemplate.Builder builder) {
        super(builder);
        this.selectors = requireNonNull(builder.selectors, "Label selector can't null");
        this.transformations = builder.transformations;
        this.predicateRules = builder.predicateRules;
    }

    public static ThresholdAlertTemplate.Builder newBuilder() {
        return new ThresholdAlertTemplate.Builder();
    }

    @Override
    public AlertTemplateType getAlertTemplateType() {
        return AlertTemplateType.THRESHOLD;
    }

    public String getSelectors() {
        return selectors;
    }

    public List<TemplatePredicateRule> getPredicateRules() {
        return predicateRules;
    }

    public String getTransformations() {
        return transformations;
    }

    @Override
    public ThresholdAlertTemplate.Builder toBuilder() {
        return new ThresholdAlertTemplate.Builder(this);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        if (!super.equals(o)) {
            return false;
        }
        ThresholdAlertTemplate that = (ThresholdAlertTemplate) o;
        return selectors.equals(that.selectors) &&
                transformations.equals(that.transformations) &&
                predicateRules.equals(that.predicateRules);
    }

    @Override
    public int hashCode() {
        return Objects.hash(super.hashCode(), selectors, transformations, predicateRules);
    }

    @Override
    public String toString() {
        return "ThresholdAlertTemplate{" +
                "selectors=" + selectors +
                ", transformations=" + transformations +
                ", predicateRules=" + predicateRules +
                "} : " + super.toString();
    }

    @ParametersAreNonnullByDefault
    public static class Builder extends AbstractAlertTemplateBuilder<ThresholdAlertTemplate, ThresholdAlertTemplate.Builder> {
        @Nullable
        private String selectors;

        private String transformations = "";
        private List<TemplatePredicateRule> predicateRules = new ArrayList<>();

        public Builder() {
        }

        public Builder(ThresholdAlertTemplate alert) {
            super(alert);
            this.selectors = alert.getSelectors();
            this.predicateRules = alert.getPredicateRules();
            this.transformations = alert.getTransformations();
        }

        @Override
        protected ThresholdAlertTemplate.Builder self() {
            return this;
        }

        public ThresholdAlertTemplate.Builder setSelectors(String selectors) {
            this.selectors = selectors;
            return self();
        }

        public ThresholdAlertTemplate.Builder setPredicateRules(Stream<TemplatePredicateRule> predicateRules) {
            this.predicateRules = predicateRules.collect(Collectors.toList());
            return self();
        }

        public ThresholdAlertTemplate.Builder setPredicateRule(TemplatePredicateRule predicateRule) {
            return setPredicateRules(Stream.of(predicateRule));
        }

        public ThresholdAlertTemplate.Builder setTransformations(String transformations) {
            this.transformations = transformations;
            return self();
        }

        @Override
        public ThresholdAlertTemplate build() {
            return new ThresholdAlertTemplate(this);
        }

    }
}
