package ru.yandex.solomon.alert.unroll;

import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import com.google.common.collect.ImmutableSet;

import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.alert.domain.expression.ExpressionAlert;
import ru.yandex.solomon.alert.rule.AlertRuleDeadlines;
import ru.yandex.solomon.alert.rule.AlertRuleSelectors;
import ru.yandex.solomon.alert.rule.ProgramCompiler;
import ru.yandex.solomon.expression.analytics.Program;
import ru.yandex.solomon.expression.compile.DeprOpts;
import ru.yandex.solomon.labels.query.Selector;
import ru.yandex.solomon.labels.query.SelectorType;
import ru.yandex.solomon.labels.query.Selectors;
import ru.yandex.solomon.metrics.client.MetricsClient;
import ru.yandex.solomon.metrics.client.UniqueLabelsRequest;

import static java.util.stream.Collectors.collectingAndThen;
import static java.util.stream.Collectors.toList;

/**
 * @author Vladimir Gordiychuk
 */
class ExpressionMultiAlertUnroll implements MultiAlertUnrollFunction {
    private final MetricsClient client;

    private final String alertId;
    private final Set<Selectors> selectors;
    private final Set<String> groupLabels;
    private final Set<String> essentialGroupKeys;

    ExpressionMultiAlertUnroll(MetricsClient client, ExpressionAlert alert) {
        if (alert.getGroupByLabels().isEmpty()) {
            throw new IllegalArgumentException("Specified alert not support unrolling: " + alert);
        }

        this.client = client;
        this.alertId = alert.getId();
        this.groupLabels = ImmutableSet.copyOf(alert.getGroupByLabels());
        List<Selectors> programSelectors = prepareProgram(alert).getProgramSelectors();

        this.essentialGroupKeys = programSelectors.stream()
                .map(this::essentialGroupKeys)
                .flatMap(Collection::stream)
                .collect(Collectors.toSet());

        this.selectors = programSelectors.stream()
                .filter(this::essentialSelectors)
                .map(s -> AlertRuleSelectors.enrichProjectSelector(alert, s))
                .collect(Collectors.toSet());
    }

    private Set<String> essentialGroupKeys(Selectors selectors) {
        // Essential group key is a key that is not fixed in the selector

        Set<String> fixed = selectors.stream()
                .filter(selector -> exactOrAbsent(selector) && groupLabels.contains(selector.getKey()))
                .map(Selector::getKey)
                .collect(Collectors.toSet());

        Set<String> essential = new HashSet<>(groupLabels);
        essential.removeAll(fixed);

        return essential;
    }

    private boolean essentialSelectors(Selectors selectors) {
        return selectors.stream()
                .noneMatch(selector -> exactOrAbsent(selector) && essentialGroupKeys.contains(selector.getKey()));
    }

    private boolean exactOrAbsent(Selector selector) {
        SelectorType type = selector.getType();
        return type == SelectorType.EXACT || type == SelectorType.ABSENT;
    }

    private Program prepareProgram(ExpressionAlert alert) {
        return Program.fromSource(ProgramCompiler.ALERTING_SEL_VERSION, alert.getCombinedSource())
                .withDeprOpts(DeprOpts.ALERTING)
                .compile();
    }

    @Override
    public CompletableFuture<UnrollResult> unroll(AlertRuleDeadlines deadlines) {
        try {
            return unsafeUnroll(deadlines);
        } catch (Throwable e) {
            return CompletableFuture.failedFuture(e);
        }
    }

    private CompletableFuture<UnrollResult> unsafeUnroll(AlertRuleDeadlines deadlines) {
        return selectors
                .stream()
                .map(selector -> resolveGroupsBySelector(deadlines, selector))
                .collect(collectingAndThen(toList(), CompletableFutures::allOf))
                .thenApply(list -> {
                    if (list.isEmpty()) {
                        return new UnrollResult(Set.of(), true);
                    }

                    if (list.size() == 1) {
                        return list.get(0);
                    }

                    Set<Labels> labels = new HashSet<>();
                    boolean allowDelete = true;
                    for (var r : list) {
                        labels.addAll(r.labels);
                        allowDelete &= r.allowDelete;
                    }
                    return new UnrollResult(labels, allowDelete);
                });
    }

    private CompletableFuture<UnrollResult> resolveGroupsBySelector(AlertRuleDeadlines deadlines, Selectors selectors) {
        UniqueLabelsRequest request = UniqueLabelsRequest.newBuilder()
                .setSelectors(selectors)
                .setSoftDeadline(deadlines.softResolveDeadline())
                .setDeadline(deadlines.hardDeadline())
                .setLabels(groupLabels)
                .build();

        return client.uniqueLabels(request)
                .thenApply(response -> {
                    if (!response.isOk()) {
                        throw new IllegalStateException(
                                "Failed resolve for alert "
                                        + alertId
                                        + " by selector "
                                        + selectors
                                        + ", metrics caused by "
                                        + response.getStatus());
                    }

                    return new UnrollResult(response.getUniqueLabels(), response.isAllDestSuccess());
                });
    }
}
