package ru.yandex.solomon.alert.unroll;

import java.util.Set;
import java.util.concurrent.CompletableFuture;

import com.google.common.collect.ImmutableSet;

import ru.yandex.solomon.alert.domain.threshold.ThresholdAlert;
import ru.yandex.solomon.alert.rule.AlertRuleDeadlines;
import ru.yandex.solomon.alert.rule.AlertRuleSelectors;
import ru.yandex.solomon.labels.query.Selectors;
import ru.yandex.solomon.metrics.client.MetricsClient;
import ru.yandex.solomon.metrics.client.UniqueLabelsRequest;

/**
 * @author Vladimir Gordiychuk
 */
class ThresholdMultiAlertUnroll implements MultiAlertUnrollFunction {
    private final MetricsClient client;

    private final String alertId;
    private final Selectors selectors;
    private final Set<String> groupLabels;

    ThresholdMultiAlertUnroll(MetricsClient client, ThresholdAlert alert) {
        this.client = client;
        this.alertId = alert.getId();
        this.selectors = AlertRuleSelectors.toOldFormat(AlertRuleSelectors.enrichProjectSelector(alert, alert.getSelectors()));
        this.groupLabels = ImmutableSet.copyOf(alert.getGroupByLabels());
    }

    @Override
    public CompletableFuture<UnrollResult> unroll(AlertRuleDeadlines deadlines) {
        try {
            return unsafeUnroll(deadlines);
        } catch (Throwable e) {
            return CompletableFuture.failedFuture(e);
        }
    }

    public CompletableFuture<UnrollResult> unsafeUnroll(AlertRuleDeadlines deadlines) {
        UniqueLabelsRequest request = UniqueLabelsRequest.newBuilder()
                .setSelectors(selectors)
                .setSoftDeadline(deadlines.softResolveDeadline())
                .setDeadline(deadlines.hardDeadline())
                .setLabels(groupLabels)
            .build();

        return client.uniqueLabels(request)
                .thenApply(response -> {
                    if (!response.isOk()) {
                        throw new IllegalStateException(
                                "Failed resolve for alert "
                                        + alertId
                                        + " by selectors "
                                        + selectors
                                        + ", metrics caused by "
                                        + response.getStatus());
                    }

                    return new UnrollResult(response.getUniqueLabels(), response.isAllDestSuccess());
                });
    }
}
