package ru.yandex.solomon.alert.unroll;

import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.monlib.metrics.histogram.Histograms;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.primitives.Histogram;
import ru.yandex.monlib.metrics.primitives.Rate;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.util.async.InFlightLimiter;


/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public class UnrollExecutorMetrics {
    private final MetricRegistry registry;
    private final ProjectMetrics total;
    private final ConcurrentMap<String, ProjectMetrics> metricsByProject = new ConcurrentHashMap<>();

    private static class ProjectMetrics {

        private final Rate tasksFailed;
        private final Histogram tasksDuration;
        ProjectMetrics(MetricRegistry registry, String projectId) {
            var labels = Labels.of("projectId", projectId);
            tasksFailed = registry.rate("unroll.tasks.failed", labels);
            tasksDuration = registry.histogramRate("unroll.tasks.duration", labels,
                    Histograms.exponential(14, 2, 16));
        }

        void finished(long elapsedMillis, boolean failed) {
            if (failed) {
                tasksFailed.inc();
            }
            tasksDuration.record(elapsedMillis);
        }
    }
    public UnrollExecutorMetrics(MetricRegistry registry) {
        this.registry = registry;
        this.total = new ProjectMetrics(registry, "total");
    }

    ProjectMetrics getMetricsForProject(String projectId) {
        var prev = metricsByProject.get(projectId);
        if (prev != null) {
            return prev;
        }
        return metricsByProject.computeIfAbsent(projectId, ignore -> new ProjectMetrics(registry, projectId));
    }

    public void unrollFinished(String projectId, long elapsedMillis, boolean failed) {
        var projectMetrics = getMetricsForProject(projectId);
        projectMetrics.finished(elapsedMillis, failed);
        total.finished(elapsedMillis, failed);
    }

    public void observe(InFlightLimiter inflightLimiter) {
        registry.lazyGaugeInt64("unroll.tasks.in_flight", inflightLimiter::getCurrent);
        registry.lazyGaugeInt64("unroll.tasks.waiting", inflightLimiter::getWaitingCount);
    }
}
