package ru.yandex.solomon.alert.util;

import java.time.Clock;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicIntegerFieldUpdater;
import java.util.concurrent.atomic.AtomicLongFieldUpdater;

/**
 * @author Vladimir Gordiychuk
 */
public class WindowRateLimit implements RateLimit {
    private static final AtomicIntegerFieldUpdater<WindowRateLimit> fieldPermits =
            AtomicIntegerFieldUpdater.newUpdater(WindowRateLimit.class, "permits");
    private static final AtomicLongFieldUpdater<WindowRateLimit> fieldLastTick =
            AtomicLongFieldUpdater.newUpdater(WindowRateLimit.class, "lastTick");

    private final Clock clock;
    private final long tickInterval;
    private int maxPermits;
    private volatile int permits;
    private volatile long lastTick;

    public WindowRateLimit(int maxPermits, TimeUnit unit) {
        this(Clock.systemUTC(), maxPermits, unit);
    }

    private long nanoTime() {
        return TimeUnit.MILLISECONDS.toNanos(clock.millis());
    }

    public WindowRateLimit(Clock clock, int maxPermits, TimeUnit unit) {
        this.clock = clock;
        this.maxPermits = maxPermits;
        this.permits = maxPermits;
        this.lastTick = nanoTime();
        this.tickInterval = unit.toNanos(1L);
    }

    @Override
    public boolean attempt() {
        tickIfNecessary();
        if (permits < 0) {
            return false;
        }

        return fieldPermits.decrementAndGet(this) >= 0;
    }

    public long delayNanosToPermit() {
        long delay = tickIfNecessary();
        if (permits < 0) {
            return delay;
        }

        if (fieldPermits.decrementAndGet(this) >= 0) {
            return 0;
        }

        return delay;
    }

    private long tickIfNecessary() {
        final long oldTick = lastTick;
        final long newTick = nanoTime();
        final long elapsed = newTick - oldTick;
        if (elapsed >= tickInterval) {
            if (fieldLastTick.compareAndSet(this, oldTick, newTick)) {
                permits = maxPermits;
            }
        }

        return (newTick - lastTick) % tickInterval;
    }
}
