package ru.yandex.solomon.alert.yachats;

import java.io.IOException;
import java.net.URI;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.time.Duration;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.stream.Collectors;

import javax.annotation.Nullable;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ObjectNode;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.commune.json.jackson.StringEnumModule;
import ru.yandex.misc.io.http.HttpStatus;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.selfmon.counters.AsyncMetrics;

public class YaChatsClientImpl implements YaChatsClient {
    private static final Logger logger = LoggerFactory.getLogger(YaChatsClient.class);

    private static final ObjectMapper mapper;
    private static final long DEFAULT_TIMEOUT_SECONDS = 15;

    static {
        mapper = new ObjectMapper();
        mapper.registerModule(new StringEnumModule());
    }

    private final Endpoint endpointSendMessage;
    private final Endpoint endpointGetChats;
    private final String oath;

    public YaChatsClientImpl(String url, String botToken, MetricRegistry reg, @Nullable Executor executor) {
        HttpClient.Builder builder = HttpClient.newBuilder()
                .followRedirects(HttpClient.Redirect.NEVER);
        if (executor != null) {
            builder.executor(executor);
        }
        this.oath = "OAuthTeam " + botToken;
        final String prefixUri = url + "/bot";
        HttpClient httpClient = builder.build();
        this.endpointSendMessage = new Endpoint(reg, prefixUri, "/sendMessage", httpClient);
        this.endpointGetChats = new Endpoint(reg, prefixUri, "/getChats", httpClient);
    }

    @Override
    public CompletableFuture<Response> sendMessage(String login, String groupId, String text) {
        ObjectNode body = mapper.createObjectNode();
        body.put("text", text);
        if (StringUtils.isNotEmpty(login)) {
            body.put("user_login", login);
        } else {
            body.put("chat_id", groupId);
        }
        body.put("disable_web_page_preview", true);

        var request = HttpRequest.newBuilder()
                .header("Content-Type", "application/json")
                .header("Authorization", oath)
                .timeout(Duration.ofSeconds(DEFAULT_TIMEOUT_SECONDS))
                .POST(HttpRequest.BodyPublishers.ofString(body.toString()));

        return endpointSendMessage.send(request).thenApply(this::processSendMessageResponse);
    }

    private Response processSendMessageResponse(HttpResponse<String> response) {
        int code = response.statusCode();
        String description = "unknown";
        if (!HttpStatus.is2xx(code)) {
            try {
                ObjectNode node = mapper.readValue(response.body(), ObjectNode.class);
                if (node.has("error") && node.get("error").isTextual()) {
                    description = node.get("error").textValue();
                }
            } catch (IOException e) {
                // just left description as unknown
            }
        }
        return new Response(code, description);
    }

    @Override
    public CompletableFuture<List<GroupsInfo>> getChats() {
        var request = HttpRequest.newBuilder()
                .header("Authorization", oath)
                .timeout(Duration.ofSeconds(DEFAULT_TIMEOUT_SECONDS))
                .GET();

        return endpointGetChats.send(request).thenApply(this::processGetChatsResponse);
    }

    private List<GroupsInfo> processGetChatsResponse(HttpResponse<String> httpResponse) {
        String response = httpResponse.body();
        if (!HttpStatus.is2xx(httpResponse.statusCode())) {
            throw new IllegalStateException(
                    "Got " + httpResponse.statusCode() + " on getCharts. Response: " + response);
        }

        try {
            List<Dto.GetChats> chats = mapper.readValue(response, new TypeReference<List<Dto.GetChats>>() {});
            return chats.stream()
                    .filter(chat -> "group".equals(chat.type) && StringUtils.isNotEmpty(chat.id))
                    .map(group -> new GroupsInfo(group.id, group.title, group.description))
                    .collect(Collectors.toList());
        } catch (Exception e) {
            throw new IllegalStateException("Can't parse JSON from: " + response, e);
        }
    }

    @Override
    public void close() {
    }

    private final static class Endpoint {
        final URI uri;
        final String endpoint;
        final AsyncMetrics callMetrics;
        final MetricRegistry reg;
        final HttpClient httpClient;

        Endpoint(MetricRegistry registry, String uriPrefix, String endpoint, HttpClient httpClient) {
            this.uri = URI.create(uriPrefix + endpoint);
            this.endpoint = uri.getHost() + endpoint;
            this.reg = registry.subRegistry("endpoint", this.endpoint);
            this.callMetrics = new AsyncMetrics(this.reg, "http.client.call");
            this.httpClient = httpClient;
        }

        void incStatus(int code) {
            reg.rate("http.client.call.status", Labels.of("status", String.valueOf(code))).inc();
        }

        public CompletableFuture<HttpResponse<String>> send(HttpRequest.Builder builder) {
            HttpRequest httpRequest = builder.uri(uri).build();
            return send(httpRequest);
        }

        private CompletableFuture<HttpResponse<String>> send(HttpRequest httpRequest) {
            return callMetrics.wrapFuture(() -> httpClient.sendAsync(httpRequest, HttpResponse.BodyHandlers.ofString()))
                    .whenComplete((response, e) -> {
                        if (e != null) {
                            logger.warn("{} failed", endpoint, e);
                            return;
                        }

                        this.incStatus(response.statusCode());
                        if (response.statusCode() != 200) {
                            logger.warn("{} response {} != 200, body {}", endpoint, response.statusCode(), response.body());
                        }
                    });
        }

        @Override
        public String toString() {
            return uri.toString();
        }
    }

}
