package ru.yandex.solomon.alert.canon;

import java.io.FileOutputStream;
import java.io.IOException;
import java.time.Instant;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;

import javax.annotation.ParametersAreNonnullByDefault;

import org.apache.commons.lang3.mutable.MutableInt;

import ru.yandex.solomon.alert.canon.protobuf.ArchiveConverterImpl;
import ru.yandex.solomon.alert.domain.Alert;
import ru.yandex.solomon.alert.protobuf.TEvaluationStatus;
import ru.yandex.solomon.alerting.canon.protobuf.TAlertEvaluationRecord;
import ru.yandex.solomon.main.logger.LoggerConfigurationUtils;
import ru.yandex.solomon.metrics.client.ReplayMetricsClient;
import ru.yandex.solomon.metricsClient.stubs.protobuf.TMetricsClientCapture;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public class RegenerateCanonResults {

    private static TAlertEvaluationRecord reevaluate(Explainer explainer, TAlertEvaluationRecord record, Instant when) {
        var explainResult = ReplayerCanonSupport.evaluate(record, explainer, when);

        Set<String> badAnnotations;
        try {
            Alert alert = ReplayerCanonSupport.extractAlert(record);
            badAnnotations = ReplayerCanonSupport.computeFlappingAnnotationKeys(alert);
        } catch (Exception e) {
            badAnnotations = Set.of();
        }

        try {
            var matcher = ResultMatcher.newBuilder()
                    .strictSeriesCheck(false)
                    .build();
            matcher.assertEqualsIgnoringStackTrace(
                    record.getExplainResult(),
                    explainResult,
                    badAnnotations);
        } catch (AssertionError e) {
            var alertKey = ReplayerCanonSupport.keyFromRecord(record);
            var check = IgnoredTests.IGNORED_KEYS.getOrDefault(alertKey, Check.FULL_CHECK);
            if (check.method() == Check.Method.IGNORED) {
                System.out.println("\n===== diff in canonical data for ignored test =====");
                System.out.println("Evaluation result changed for " + alertKey);
                System.out.println(e.getMessage());
                System.out.printf("This is JFYI, the test is ignored, reason: %s\n", check.explanation());
            } else {
                System.err.println("\n===== !!! DIFF IN CANONICAL DATA !!! =====");
                System.err.println("Evaluation result changed for " + alertKey);
                System.err.println(e.getMessage());
                System.err.println("Ignore this error if the change was intended");
            }
        }

        return record.toBuilder()
                .setExplainResult(explainResult)
                .build();
    }

    static Serialization.CanonResults makeNewCanonResults(
            TMetricsClientCapture metricsClientCapture,
            Serialization.CanonResults oldResults)
    {
        ReplayMetricsClient replayMetricsClient = new ReplayMetricsClient(metricsClientCapture, ArchiveConverterImpl.I, true);
        Explainer explainer = new Explainer(replayMetricsClient);

        var newResults = new Serialization.CanonResults();

        newResults.evaluationMoment = oldResults.evaluationMoment;
        newResults.alertEvaluationRecords = new ArrayList<>();

        Map<TEvaluationStatus.ECode, MutableInt> countByStatus = new HashMap<>();

        for (var record : oldResults.alertEvaluationRecords) {
            var newRecord = reevaluate(explainer, record, oldResults.evaluationMoment);
            newResults.alertEvaluationRecords.add(newRecord);
            countByStatus.computeIfAbsent(newRecord.getExplainResult().getEvaluationStatus().getCode(),
                    ignore -> new MutableInt(0)).increment();
        }

        System.out.println("Test set stats: " + countByStatus);
        System.err.println("\nPlease review the log above CAREFULLY. Check that every diff is either expected or already ignored");

        return newResults;
    }

    /**
     * Generate fresh data from same sources
     */
    public static void main(String[] args) throws IOException {
        LoggerConfigurationUtils.disableLogger();
        String prefix = "classpath:"; // "classpath:" - jar from SB, "/tmp/" - from disk

        var metricsClientCapture = ReplayerCanonSupport.readMetricsClientCapture(prefix);
        var canonicalResults = ReplayerCanonSupport.readCanonicalResults(prefix);

        ZipOutputStream jar = new ZipOutputStream(new FileOutputStream("/tmp/alert_canon_data.jar"));

        jar.putNextEntry(new ZipEntry(ReplayerCanonSupport.METRICS_CLIENT_CAPTURE));
        Serialization.writeMetricsClientCapture(metricsClientCapture, jar);
        jar.closeEntry();

        var freshCanonResults = makeNewCanonResults(metricsClientCapture, canonicalResults);

        jar.putNextEntry(new ZipEntry(ReplayerCanonSupport.CANONICAL_EVALUATION_RESULT));
        Serialization.writeCanonicalResults(freshCanonResults, jar);
        jar.closeEntry();

        jar.close();
    }
}
